package com.aliyun.sdk.gateway.pop.auth.signer;

import com.aliyun.auth.signature.SignerParams;
import com.aliyun.auth.signature.signer.SignAlgorithmHmacSHA256;
import com.aliyun.auth.signature.signer.SignAlgorithmHmacSM3;
import com.aliyun.auth.signature.signer.SignAlgorithmSHA256withRSA;
import com.aliyun.sdk.gateway.pop.auth.SignatureAlgorithm;
import com.aliyun.sdk.gateway.pop.auth.SignatureVersion;
import com.aliyun.auth.credentials.ICredential;
import org.bouncycastle.crypto.macs.HMac;
import org.bouncycastle.crypto.params.KeyParameter;
import org.bouncycastle.jce.provider.BouncyCastleProvider;

import javax.crypto.Mac;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import java.io.UnsupportedEncodingException;
import java.security.*;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.Base64;

import static com.aliyun.core.utils.EncodeUtil.hexEncode;

public class PopV3Signer implements PopSigner {
    private final String ENCODING = "UTF-8";
    private final String HASH_SHA256 = "SHA-256";
    private final String HASH_SM3 = "SM3";
    private final String PEM_BEGIN = "-----BEGIN RSA PRIVATE KEY-----\n";
    private final String PEM_END = "\n-----END RSA PRIVATE KEY-----";
    private final SignatureAlgorithm DEFAULT_ALGORITHM = SignatureAlgorithm.ACS3_HMAC_SHA256;
    private final SignatureAlgorithm algorithm;

    public PopV3Signer() {
        this.algorithm = DEFAULT_ALGORITHM;
    }

    public PopV3Signer(SignatureAlgorithm algorithm) {
        if (algorithm != null) {
            this.algorithm = algorithm;
        } else {
            this.algorithm = DEFAULT_ALGORITHM;
        }
    }

    @Override
    public String signString(String stringToSign, SignerParams params) {
        ICredential credential = params.credentials();
        String accessKeySecret = credential.accessKeySecret();
        try {
            switch (algorithm) {
                case ACS3_HMAC_SM3:
                    HMac sm3HMAC = SignAlgorithmHmacSM3.HMAC_SM3.getMac();
                    SecretKey key = new SecretKeySpec(accessKeySecret.getBytes(ENCODING), SignAlgorithmHmacSM3.HMAC_SM3.toString());
                    byte[] sm3Sign = new byte[sm3HMAC.getMacSize()];
                    byte[] inputBytes = stringToSign.getBytes(ENCODING);
                    sm3HMAC.init(new KeyParameter(key.getEncoded()));
                    sm3HMAC.update(inputBytes, 0, inputBytes.length);
                    sm3HMAC.doFinal(sm3Sign, 0);
                    return hexEncode(sm3Sign);
                case ACS3_RSA_SHA256:
                    Signature rsaSign = SignAlgorithmSHA256withRSA.SHA256withRSA.getSignature();
                    KeyFactory kf = KeyFactory.getInstance(SignAlgorithmSHA256withRSA.SHA256withRSA.toString());
                    byte[] keySpec = Base64.getDecoder().decode(checkRSASecret(accessKeySecret));
                    PrivateKey privateKey = kf.generatePrivate(new PKCS8EncodedKeySpec(keySpec));
                    rsaSign.initSign(privateKey);
                    rsaSign.update(stringToSign.getBytes(ENCODING));
                    byte[] sign = rsaSign.sign();
                    return hexEncode(sign);
                case ACS3_HMAC_SHA256:
                default:
                    Mac sha256HMAC = SignAlgorithmHmacSHA256.HmacSHA256.getMac();
                    sha256HMAC.init(new SecretKeySpec(accessKeySecret.getBytes(ENCODING), SignAlgorithmHmacSHA256.HmacSHA256.toString()));
                    return hexEncode(sha256HMAC.doFinal(stringToSign.getBytes(ENCODING)));
            }
        } catch (UnsupportedEncodingException | NoSuchAlgorithmException | InvalidKeySpecException | InvalidKeyException | SignatureException e) {
            throw new com.aliyun.auth.signature.exception.SignatureException(e.toString());
        }

    }

    private String checkRSASecret(String accessKeySecret) {
        if (accessKeySecret != null) {
            if (accessKeySecret.startsWith(PEM_BEGIN)) {
                accessKeySecret = accessKeySecret.replace(PEM_BEGIN, "");
            }
            while (accessKeySecret.endsWith("\n") || accessKeySecret.endsWith("\r")) {
                accessKeySecret = accessKeySecret.substring(0, accessKeySecret.length() - 1);
            }
            if (accessKeySecret.endsWith(PEM_END)) {
                accessKeySecret = accessKeySecret.replace(PEM_END, "");
            }
        }
        return accessKeySecret;
    }

    @Override
    public byte[] hash(byte[] raw) {
        if (null == raw) {
            return null;
        }
        try {
            MessageDigest digest;
            switch (algorithm) {
                case ACS3_HMAC_SM3:
                    BouncyCastleProvider provider = new BouncyCastleProvider();
                    digest = MessageDigest.getInstance(HASH_SM3, provider);
                    break;
                case ACS3_HMAC_SHA256:
                case ACS3_RSA_SHA256:
                default:
                    digest = MessageDigest.getInstance(HASH_SHA256);
            }
            return digest.digest(raw);
        } catch (NoSuchAlgorithmException e) {
            throw new com.aliyun.auth.signature.exception.SignatureException(e.toString());
        }
    }

    @Override
    public String getContent() {
        switch (algorithm) {
            case ACS3_HMAC_SM3:
                return "x-acs-content-sm3";
            case ACS3_HMAC_SHA256:
            case ACS3_RSA_SHA256:
            default:
                return "x-acs-content-sha256";
        }
    }

    @Override
    public String getSignerName() {
        return algorithm.algorithmName();
    }

    @Override
    public SignatureVersion getSignerVersion() {
        return SignatureVersion.V3;
    }
}
