package darabonba.core.async;

import org.reactivestreams.Publisher;
import org.reactivestreams.Subscriber;
import org.reactivestreams.Subscription;

import java.io.*;
import java.nio.ByteBuffer;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;

import static com.aliyun.core.utils.FunctionalUtils.invokeSafely;

public class FileAsyncResponseHandler<ResponseT> implements AsyncResponseHandler<ResponseT, ResponseT> {
    private final Path path;
    private FileOutputStream fileOutputStream;
    private FileSubscriber subscriber;

    public FileAsyncResponseHandler(Path path) {
        this.path = path;
    }

    private FileOutputStream createFileOutputStream(Path path) throws IOException {
        return new FileOutputStream(new File(path.toUri()));
    }

    @Override
    public void onStream(Publisher<ByteBuffer> publisher) {
        this.fileOutputStream = invokeSafely(() -> createFileOutputStream(path));
        this.subscriber = new FileSubscriber(this.fileOutputStream, path, this::exceptionOccurred);
        publisher.subscribe(this.subscriber);
    }

    @Override
    public void onError(Throwable throwable) {
        if (this.subscriber != null) {
            this.subscriber.onError(throwable);
        }
    }

    @Override
    public ResponseT transform(ResponseT response) {
        return response;
    }

    private void exceptionOccurred(Throwable throwable) {
        try {
            if (fileOutputStream != null) {
                invokeSafely(fileOutputStream::close);
            }
        } finally {
            invokeSafely(() -> Files.deleteIfExists(path));
        }
    }

    static class FileSubscriber implements Subscriber<ByteBuffer> {
        private final AtomicLong position = new AtomicLong();

        private final Path path;
        private final Consumer<Throwable> onErrorMethod;
        private final FileOutputStream fileOutputStream;
        private final OutputStream outputStream;

        private volatile boolean writeInProgress = false;
        private volatile boolean closeOnLastWrite = false;
        private Subscription subscription;

        FileSubscriber(FileOutputStream fileOutputStream, Path path, Consumer<Throwable> onErrorMethod) {
            this.fileOutputStream = fileOutputStream;
            this.outputStream = new BufferedOutputStream(fileOutputStream);
            this.path = path;
            this.onErrorMethod = onErrorMethod;
        }

        @Override
        public void onSubscribe(Subscription s) {
            if (this.subscription != null) {
                s.cancel();
                return;
            }
            this.subscription = s;
            // Request the first chunk to start producing content
            s.request(1);
        }

        @Override
        public void onNext(ByteBuffer byteBuffer) {
            if (byteBuffer == null) {
                throw new NullPointerException("Element must not be null");
            }

            if (outputStream != null) {
                performWrite(byteBuffer);
            }
        }

        private void performWrite(ByteBuffer byteBuffer) {
            byte[] b = byteBuffer.array();
            int off = byteBuffer.arrayOffset() + byteBuffer.position();
            int len = byteBuffer.remaining();
            invokeSafely(() -> outputStream.write(b, off, len));
            subscription.request(1);
        }

        @Override
        public void onError(Throwable t) {
            onErrorMethod.accept(t);
        }

        @Override
        public void onComplete() {
            // if write in progress, tell write to close on finish.
            close();
        }

        private void close() {
            if (outputStream != null) {
                invokeSafely(outputStream::close);
            }
        }

        @Override
        public String toString() {
            return getClass() + ":" + path.toString();
        }
    }
}
