/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.servlet;

import java.util.List;

import javax.annotation.Nonnull;
import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.annotation.WebFilter;
import javax.servlet.http.HttpServletRequest;

import com.sap.cloud.sdk.cloudplatform.exception.ShouldNotHappenException;
import com.sap.cloud.sdk.cloudplatform.thread.ThreadContext;
import com.sap.cloud.sdk.cloudplatform.thread.ThreadContextExecutor;
import com.sap.cloud.sdk.cloudplatform.thread.ThreadContextListener;

import io.vavr.control.Option;
import lombok.extern.slf4j.Slf4j;

/**
 * Servlet filter for storing the current {@link HttpServletRequest} in the current thread context.
 */
@WebFilter( filterName = "RequestAccessorFilter", urlPatterns = "/*" )
@Slf4j
public class RequestAccessorFilter implements Filter
{
    @Override
    public void init( @Nonnull final FilterConfig filterConfig )
    {

    }

    @Override
    public void doFilter(
        @Nonnull final ServletRequest request,
        @Nonnull final ServletResponse response,
        @Nonnull final FilterChain filterChain )
    {
        if( request instanceof HttpServletRequest ) {
            try {
                final ThreadContextExecutor threadContextExecutor = new ThreadContextExecutor();

                final List<ThreadContextListener> listeners = threadContextExecutor.getListenersOrderedByPriority();

                final Option<ThreadContextListener> existing =
                    Option.ofOptional(
                        listeners
                            .stream()
                            .filter(
                                listener -> listener
                                    .getPriority() == ThreadContextListener.DefaultPriorities.REQUEST_LISTENER)
                            .findAny());

                if( !existing.isDefined() ) {
                    threadContextExecutor.withListeners(new RequestThreadContextListener((HttpServletRequest) request));
                } else {
                    if( log.isWarnEnabled() ) {
                        log.warn(
                            "Failed to add "
                                + ThreadContextListener.class.getSimpleName()
                                + ": listener "
                                + existing.get().getClass().getName()
                                + " with priority "
                                + ThreadContextListener.DefaultPriorities.REQUEST_LISTENER
                                + " already exists.");
                    }
                }

                threadContextExecutor.execute(() -> filterChain.doFilter(request, response));
            }
            catch( final Throwable t ) { // ALLOW CATCH THROWABLE
                log.warn("Unexpected servlet filter exception: " + t.getMessage(), t);
                throw new ShouldNotHappenException(t);
            }
        } else {
            if( log.isWarnEnabled() ) {
                log.warn(
                    "Failed to initialize "
                        + ThreadContext.class.getSimpleName()
                        + ": request not of type "
                        + HttpServletRequest.class.getName()
                        + ".");
            }
        }
    }

    @Override
    public void destroy()
    {

    }
}
