/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.servlet;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.servlet.http.HttpServletRequest;

import com.sap.cloud.sdk.cloudplatform.thread.Property;
import com.sap.cloud.sdk.cloudplatform.thread.ThreadContext;
import com.sap.cloud.sdk.cloudplatform.thread.ThreadContextListener;

import io.vavr.control.Try;
import lombok.RequiredArgsConstructor;

/**
 * Implementation of {@link ThreadContextListener} that ensures that the current
 * {@link javax.servlet.http.HttpServletRequest}
 */
@RequiredArgsConstructor
public class RequestThreadContextListener implements ThreadContextListener
{
    /**
     * The ThreadContext key.
     */
    public static final String PROPERTY_REQUEST = RequestThreadContextListener.class.getName() + ":request";

    /**
     * The {@link HttpServletRequest} to be used by this listener.
     */
    @Nonnull
    private final HttpServletRequest request;

    /**
     * {@inheritDoc}
     */
    @Override
    public int getPriority()
    {
        return DefaultPriorities.REQUEST_LISTENER;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void beforeInitialize(
        @Nonnull final ThreadContext threadContext,
        @Nullable final ThreadContext parentThreadContext )
    {
        threadContext.setPropertyIfAbsent(PROPERTY_REQUEST, Try.of(() -> Property.of(request)));
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public
        void
        beforeDestroy( @Nonnull final ThreadContext threadContext, @Nullable final ThreadContext parentThreadContext )
    {
        threadContext.removeProperty(PROPERTY_REQUEST);
    }
}
