/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import javax.annotation.Nonnull;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.bank.Bank;

/**
 * <p>
 * 
 * You can use this inbound synchronous service to create, read, update, and mark bank master data for deletion. Use a
 * standard archiving tool for the actual deletion of bank master data. The service is based on the OData protocol and
 * can be consumed in SAP Fiori apps and other user interfaces. The service contains Bank Country, Bank Internal ID,
 * Bank Name, Region, Street Name, City Name, SWIFT Code, Bank Network Grouping, Is Marked For Deletion, Bank Branch and
 * Bank Category..
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/55/12cf2c6e2e45c8a141581ddd708647/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference:
 * <a href='https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/BANK_0001?section=OVERVIEW'>SAP API
 * Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>BANK_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Bank - Bank Integration (SAP_COM_0379)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/BFA'>Basic Bank Account Management (BFA)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509, OAuth2</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_BANK</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>Bank</td>
 * </tr>
 * </table>
 * 
 */
public interface BankService
{

    /**
     * If no other path was provided via the {@link #withServicePath(String)} method, this is the default service path
     * used to access the endpoint.
     * 
     */
    String DEFAULT_SERVICE_PATH = "/sap/opu/odata4/sap/api_bank/srvd_a2x/sap/bank/0001";

    /**
     * Overrides the default service path and returns a new service instance with the specified service path. Also
     * adjusts the respective entity URLs.
     * 
     * @param servicePath
     *            Service path that will override the default.
     * @return A new service instance with the specified service path.
     */
    @Nonnull
    BankService withServicePath( @Nonnull final String servicePath );

    /**
     * Creates a batch request builder object.
     * 
     * @return A request builder to handle batch operation on this service. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder#execute(HttpDestinationProperties)
     *         execute} method on the request builder object.
     */
    @Nonnull
    BatchRequestBuilder batch();

    /**
     * Fetch multiple {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.bank.Bank Bank} entities.
     * 
     * @return A request builder to fetch multiple
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.bank.Bank Bank} entities. This request
     *         builder allows methods which modify the underlying query to be called before executing the query itself.
     *         To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.bank.Bank>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetAllRequestBuilder<Bank> getAllBank();

    /**
     * Fetch the number of entries from the {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.bank.Bank Bank}
     * entity collection matching the filter and search expressions.
     * 
     * @return A request builder to fetch the count of
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.bank.Bank Bank} entities. This request
     *         builder allows methods which modify the underlying query to be called before executing the query itself.
     *         To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.bank.Bank>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CountRequestBuilder<Bank> countBank();

    /**
     * Fetch a single {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.bank.Bank Bank} entity using key
     * fields.
     * 
     * @param bankInternalID
     *            Bank Keys
     *            <p>
     *            Constraints: Not nullable, Maximum length: 15
     *            </p>
     * @param bankCountry
     *            Country/Region Key of Bank
     *            <p>
     *            Constraints: Not nullable, Maximum length: 3
     *            </p>
     * @return A request builder to fetch a single
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.bank.Bank Bank} entity using key fields.
     *         This request builder allows methods which modify the underlying query to be called before executing the
     *         query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.bank.Bank>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetByKeyRequestBuilder<Bank>
        getBankByKey( @Nonnull final String bankCountry, @Nonnull final String bankInternalID );

    /**
     * Create a new {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.bank.Bank Bank} entity and save it to
     * the S/4HANA system.
     * 
     * @param bank
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.bank.Bank Bank} entity object that will
     *            be created in the S/4HANA system.
     * @return A request builder to create a new {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.bank.Bank
     *         Bank} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.bank.Bank>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CreateRequestBuilder<Bank> createBank( @Nonnull final Bank bank );

    /**
     * Update an existing {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.bank.Bank Bank} entity and save
     * it to the S/4HANA system.
     * 
     * @param bank
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.bank.Bank Bank} entity object that will
     *            be updated in the S/4HANA system.
     * @return A request builder to update an existing
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.bank.Bank Bank} entity. To perform
     *         execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.bank.Bank>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    UpdateRequestBuilder<Bank> updateBank( @Nonnull final Bank bank );

}
