/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.time.LocalDate;
import java.util.UUID;

import javax.annotation.Nonnull;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.DeleteRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.BusVolSelectionCriteria;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrCndnRecordValidity;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecord;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecordScale;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrSettlementCalendar;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContract;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContractStatus;

/**
 * <p>
 * 
 * The service enables you to retrieve and process transactional data of Condition Contacts:
 * 
 * You can read, create and update Condition Contracts.
 * 
 * You can also release Condition Contracts, as well as lock and release them for settlement.
 * 
 * You can determine the accruals amounts for a specific Condition Contract and group them based on several aspects.
 * 
 * You can read and create new entries in the business volume selection criteria of the Condition Contract. Furthermore,
 * you can also update and delete the entries.
 * 
 * You can read and create new entries in the settlement calendar of the Condition Contract. Furthermore, you can also
 * update and delete the entries.
 * 
 * You can read the status of the Condition Contract.
 * 
 * You can read and create new condition records. Furthermore, you can update and delete them. It is also possible to
 * set a condition record as invalid and to revoke the invalidity.
 * 
 * You can read and change the validity of the condition records.
 * 
 * You can read and create new condition record scales in a condition record. It is also possible to delete them..
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/43/a9ffe7477f47c3958930b832ea52c5/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/CONDITIONCONTRACT_0001?section=OVERVIEW'>SAP
 * API Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>CONDITIONCONTRACT_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Logistics - Condition Contract Integration (SAP_COM_0729)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/1B6'>Sales Rebate Processing (1B6)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/2R7'>Purchasing Rebate Processing (2R7)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/2TT'>Sales Commissions - External Sales Representative (2TT)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/55I'>Royalties Settlement for Sales (55I)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_CONDITION_CONTRACT</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>ConditionContract</td>
 * </tr>
 * </table>
 * 
 */
public interface ConditionContractService
{

    /**
     * If no other path was provided via the {@link #withServicePath(String)} method, this is the default service path
     * used to access the endpoint.
     * 
     */
    String DEFAULT_SERVICE_PATH = "/sap/opu/odata4/sap/api_condition_contract/srvd_a2x/sap/conditioncontract/0001";

    /**
     * Overrides the default service path and returns a new service instance with the specified service path. Also
     * adjusts the respective entity URLs.
     * 
     * @param servicePath
     *            Service path that will override the default.
     * @return A new service instance with the specified service path.
     */
    @Nonnull
    ConditionContractService withServicePath( @Nonnull final String servicePath );

    /**
     * Creates a batch request builder object.
     * 
     * @return A request builder to handle batch operation on this service. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder#execute(HttpDestinationProperties)
     *         execute} method on the request builder object.
     */
    @Nonnull
    BatchRequestBuilder batch();

    /**
     * Fetch multiple
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.BusVolSelectionCriteria
     * BusVolSelectionCriteria} entities.
     * 
     * @return A request builder to fetch multiple
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.BusVolSelectionCriteria
     *         BusVolSelectionCriteria} entities. This request builder allows methods which modify the underlying query
     *         to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.BusVolSelectionCriteria>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetAllRequestBuilder<BusVolSelectionCriteria> getAllBusVolSelectionCriteria();

    /**
     * Fetch the number of entries from the
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.BusVolSelectionCriteria
     * BusVolSelectionCriteria} entity collection matching the filter and search expressions.
     * 
     * @return A request builder to fetch the count of
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.BusVolSelectionCriteria
     *         BusVolSelectionCriteria} entities. This request builder allows methods which modify the underlying query
     *         to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.BusVolSelectionCriteria>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CountRequestBuilder<BusVolSelectionCriteria> countBusVolSelectionCriteria();

    /**
     * Fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.BusVolSelectionCriteria
     * BusVolSelectionCriteria} entity using key fields.
     * 
     * @param cndnContrBusVolUUID
     *            Condition Contract: Business volume base item GUID
     *            <p>
     *            Constraints: Not nullable
     *            </p>
     * @param conditionContract
     *            Condition Contract
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @return A request builder to fetch a single
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.BusVolSelectionCriteria
     *         BusVolSelectionCriteria} entity using key fields. This request builder allows methods which modify the
     *         underlying query to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.BusVolSelectionCriteria>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetByKeyRequestBuilder<BusVolSelectionCriteria> getBusVolSelectionCriteriaByKey(
        @Nonnull final String conditionContract,
        @Nonnull final UUID cndnContrBusVolUUID );

    /**
     * Create a new
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.BusVolSelectionCriteria
     * BusVolSelectionCriteria} entity and save it to the S/4HANA system.
     * 
     * @param busVolSelectionCriteria
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.BusVolSelectionCriteria
     *            BusVolSelectionCriteria} entity object that will be created in the S/4HANA system.
     * @return A request builder to create a new
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.BusVolSelectionCriteria
     *         BusVolSelectionCriteria} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.BusVolSelectionCriteria>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CreateRequestBuilder<BusVolSelectionCriteria>
        createBusVolSelectionCriteria( @Nonnull final BusVolSelectionCriteria busVolSelectionCriteria );

    /**
     * Update an existing
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.BusVolSelectionCriteria
     * BusVolSelectionCriteria} entity and save it to the S/4HANA system.
     * 
     * @param busVolSelectionCriteria
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.BusVolSelectionCriteria
     *            BusVolSelectionCriteria} entity object that will be updated in the S/4HANA system.
     * @return A request builder to update an existing
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.BusVolSelectionCriteria
     *         BusVolSelectionCriteria} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.BusVolSelectionCriteria>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    UpdateRequestBuilder<BusVolSelectionCriteria>
        updateBusVolSelectionCriteria( @Nonnull final BusVolSelectionCriteria busVolSelectionCriteria );

    /**
     * Deletes an existing
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.BusVolSelectionCriteria
     * BusVolSelectionCriteria} entity in the S/4HANA system.
     * 
     * @param busVolSelectionCriteria
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.BusVolSelectionCriteria
     *            BusVolSelectionCriteria} entity object that will be deleted in the S/4HANA system.
     * @return A request builder to delete an existing
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.BusVolSelectionCriteria
     *         BusVolSelectionCriteria} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.DeleteRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.BusVolSelectionCriteria>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    DeleteRequestBuilder<BusVolSelectionCriteria>
        deleteBusVolSelectionCriteria( @Nonnull final BusVolSelectionCriteria busVolSelectionCriteria );

    /**
     * Fetch multiple
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrCndnRecordValidity
     * CndnContrCndnRecordValidity} entities.
     * 
     * @return A request builder to fetch multiple
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrCndnRecordValidity
     *         CndnContrCndnRecordValidity} entities. This request builder allows methods which modify the underlying
     *         query to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrCndnRecordValidity>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetAllRequestBuilder<CndnContrCndnRecordValidity> getAllCndnContrCndnRecordValidity();

    /**
     * Fetch the number of entries from the
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrCndnRecordValidity
     * CndnContrCndnRecordValidity} entity collection matching the filter and search expressions.
     * 
     * @return A request builder to fetch the count of
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrCndnRecordValidity
     *         CndnContrCndnRecordValidity} entities. This request builder allows methods which modify the underlying
     *         query to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrCndnRecordValidity>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CountRequestBuilder<CndnContrCndnRecordValidity> countCndnContrCndnRecordValidity();

    /**
     * Fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrCndnRecordValidity
     * CndnContrCndnRecordValidity} entity using key fields.
     * 
     * @param conditionValidityEndDate
     *            Validity end date of the condition record
     *            <p>
     *            Constraints: Not nullable
     *            </p>
     * @param conditionRecord
     *            Number of the Condition Record
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param conditionContract
     *            Condition Contract
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @return A request builder to fetch a single
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrCndnRecordValidity
     *         CndnContrCndnRecordValidity} entity using key fields. This request builder allows methods which modify
     *         the underlying query to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrCndnRecordValidity>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetByKeyRequestBuilder<CndnContrCndnRecordValidity> getCndnContrCndnRecordValidityByKey(
        @Nonnull final String conditionContract,
        @Nonnull final String conditionRecord,
        @Nonnull final LocalDate conditionValidityEndDate );

    /**
     * Create a new
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrCndnRecordValidity
     * CndnContrCndnRecordValidity} entity and save it to the S/4HANA system.
     * 
     * @param cndnContrCndnRecordValidity
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrCndnRecordValidity
     *            CndnContrCndnRecordValidity} entity object that will be created in the S/4HANA system.
     * @return A request builder to create a new
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrCndnRecordValidity
     *         CndnContrCndnRecordValidity} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrCndnRecordValidity>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CreateRequestBuilder<CndnContrCndnRecordValidity>
        createCndnContrCndnRecordValidity( @Nonnull final CndnContrCndnRecordValidity cndnContrCndnRecordValidity );

    /**
     * Fetch multiple
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecord
     * CndnContrConditionRecord} entities.
     * 
     * @return A request builder to fetch multiple
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecord
     *         CndnContrConditionRecord} entities. This request builder allows methods which modify the underlying query
     *         to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecord>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetAllRequestBuilder<CndnContrConditionRecord> getAllCndnContrConditionRecord();

    /**
     * Fetch the number of entries from the
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecord
     * CndnContrConditionRecord} entity collection matching the filter and search expressions.
     * 
     * @return A request builder to fetch the count of
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecord
     *         CndnContrConditionRecord} entities. This request builder allows methods which modify the underlying query
     *         to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecord>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CountRequestBuilder<CndnContrConditionRecord> countCndnContrConditionRecord();

    /**
     * Fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecord
     * CndnContrConditionRecord} entity using key fields.
     * 
     * @param conditionRecord
     *            Number of the Condition Record
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param conditionContract
     *            Condition Contract
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @return A request builder to fetch a single
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecord
     *         CndnContrConditionRecord} entity using key fields. This request builder allows methods which modify the
     *         underlying query to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecord>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetByKeyRequestBuilder<CndnContrConditionRecord> getCndnContrConditionRecordByKey(
        @Nonnull final String conditionContract,
        @Nonnull final String conditionRecord );

    /**
     * Create a new
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecord
     * CndnContrConditionRecord} entity and save it to the S/4HANA system.
     * 
     * @param cndnContrConditionRecord
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecord
     *            CndnContrConditionRecord} entity object that will be created in the S/4HANA system.
     * @return A request builder to create a new
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecord
     *         CndnContrConditionRecord} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecord>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CreateRequestBuilder<CndnContrConditionRecord>
        createCndnContrConditionRecord( @Nonnull final CndnContrConditionRecord cndnContrConditionRecord );

    /**
     * Update an existing
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecord
     * CndnContrConditionRecord} entity and save it to the S/4HANA system.
     * 
     * @param cndnContrConditionRecord
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecord
     *            CndnContrConditionRecord} entity object that will be updated in the S/4HANA system.
     * @return A request builder to update an existing
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecord
     *         CndnContrConditionRecord} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecord>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    UpdateRequestBuilder<CndnContrConditionRecord>
        updateCndnContrConditionRecord( @Nonnull final CndnContrConditionRecord cndnContrConditionRecord );

    /**
     * Deletes an existing
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecord
     * CndnContrConditionRecord} entity in the S/4HANA system.
     * 
     * @param cndnContrConditionRecord
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecord
     *            CndnContrConditionRecord} entity object that will be deleted in the S/4HANA system.
     * @return A request builder to delete an existing
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecord
     *         CndnContrConditionRecord} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.DeleteRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecord>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    DeleteRequestBuilder<CndnContrConditionRecord>
        deleteCndnContrConditionRecord( @Nonnull final CndnContrConditionRecord cndnContrConditionRecord );

    /**
     * Fetch multiple
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecordScale
     * CndnContrConditionRecordScale} entities.
     * 
     * @return A request builder to fetch multiple
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecordScale
     *         CndnContrConditionRecordScale} entities. This request builder allows methods which modify the underlying
     *         query to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecordScale>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetAllRequestBuilder<CndnContrConditionRecordScale> getAllCndnContrConditionRecordScale();

    /**
     * Fetch the number of entries from the
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecordScale
     * CndnContrConditionRecordScale} entity collection matching the filter and search expressions.
     * 
     * @return A request builder to fetch the count of
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecordScale
     *         CndnContrConditionRecordScale} entities. This request builder allows methods which modify the underlying
     *         query to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecordScale>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CountRequestBuilder<CndnContrConditionRecordScale> countCndnContrConditionRecordScale();

    /**
     * Fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecordScale
     * CndnContrConditionRecordScale} entity using key fields.
     * 
     * @param conditionRecord
     *            Number of the Condition Record
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param conditionScaleLine
     *            Current number of the line scale
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param conditionContract
     *            Condition Contract
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @return A request builder to fetch a single
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecordScale
     *         CndnContrConditionRecordScale} entity using key fields. This request builder allows methods which modify
     *         the underlying query to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecordScale>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetByKeyRequestBuilder<CndnContrConditionRecordScale> getCndnContrConditionRecordScaleByKey(
        @Nonnull final String conditionContract,
        @Nonnull final String conditionRecord,
        @Nonnull final String conditionScaleLine );

    /**
     * Create a new
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecordScale
     * CndnContrConditionRecordScale} entity and save it to the S/4HANA system.
     * 
     * @param cndnContrConditionRecordScale
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecordScale
     *            CndnContrConditionRecordScale} entity object that will be created in the S/4HANA system.
     * @return A request builder to create a new
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecordScale
     *         CndnContrConditionRecordScale} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecordScale>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CreateRequestBuilder<CndnContrConditionRecordScale> createCndnContrConditionRecordScale(
        @Nonnull final CndnContrConditionRecordScale cndnContrConditionRecordScale );

    /**
     * Deletes an existing
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecordScale
     * CndnContrConditionRecordScale} entity in the S/4HANA system.
     * 
     * @param cndnContrConditionRecordScale
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecordScale
     *            CndnContrConditionRecordScale} entity object that will be deleted in the S/4HANA system.
     * @return A request builder to delete an existing
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecordScale
     *         CndnContrConditionRecordScale} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.DeleteRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecordScale>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    DeleteRequestBuilder<CndnContrConditionRecordScale> deleteCndnContrConditionRecordScale(
        @Nonnull final CndnContrConditionRecordScale cndnContrConditionRecordScale );

    /**
     * Fetch multiple
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrSettlementCalendar
     * CndnContrSettlementCalendar} entities.
     * 
     * @return A request builder to fetch multiple
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrSettlementCalendar
     *         CndnContrSettlementCalendar} entities. This request builder allows methods which modify the underlying
     *         query to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrSettlementCalendar>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetAllRequestBuilder<CndnContrSettlementCalendar> getAllCndnContrSettlementCalendar();

    /**
     * Fetch the number of entries from the
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrSettlementCalendar
     * CndnContrSettlementCalendar} entity collection matching the filter and search expressions.
     * 
     * @return A request builder to fetch the count of
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrSettlementCalendar
     *         CndnContrSettlementCalendar} entities. This request builder allows methods which modify the underlying
     *         query to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrSettlementCalendar>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CountRequestBuilder<CndnContrSettlementCalendar> countCndnContrSettlementCalendar();

    /**
     * Fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrSettlementCalendar
     * CndnContrSettlementCalendar} entity using key fields.
     * 
     * @param settlmtDate
     *            Rebates: Settlement Date
     *            <p>
     *            Constraints: Not nullable
     *            </p>
     * @param conditionContract
     *            Condition Contract
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param settlmtDateCat
     *            Rebates: Settlement Date Type
     *            <p>
     *            Constraints: Not nullable, Maximum length: 1
     *            </p>
     * @return A request builder to fetch a single
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrSettlementCalendar
     *         CndnContrSettlementCalendar} entity using key fields. This request builder allows methods which modify
     *         the underlying query to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrSettlementCalendar>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetByKeyRequestBuilder<CndnContrSettlementCalendar> getCndnContrSettlementCalendarByKey(
        @Nonnull final String conditionContract,
        @Nonnull final LocalDate settlmtDate,
        @Nonnull final String settlmtDateCat );

    /**
     * Create a new
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrSettlementCalendar
     * CndnContrSettlementCalendar} entity and save it to the S/4HANA system.
     * 
     * @param cndnContrSettlementCalendar
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrSettlementCalendar
     *            CndnContrSettlementCalendar} entity object that will be created in the S/4HANA system.
     * @return A request builder to create a new
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrSettlementCalendar
     *         CndnContrSettlementCalendar} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrSettlementCalendar>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CreateRequestBuilder<CndnContrSettlementCalendar>
        createCndnContrSettlementCalendar( @Nonnull final CndnContrSettlementCalendar cndnContrSettlementCalendar );

    /**
     * Update an existing
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrSettlementCalendar
     * CndnContrSettlementCalendar} entity and save it to the S/4HANA system.
     * 
     * @param cndnContrSettlementCalendar
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrSettlementCalendar
     *            CndnContrSettlementCalendar} entity object that will be updated in the S/4HANA system.
     * @return A request builder to update an existing
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrSettlementCalendar
     *         CndnContrSettlementCalendar} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrSettlementCalendar>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    UpdateRequestBuilder<CndnContrSettlementCalendar>
        updateCndnContrSettlementCalendar( @Nonnull final CndnContrSettlementCalendar cndnContrSettlementCalendar );

    /**
     * Deletes an existing
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrSettlementCalendar
     * CndnContrSettlementCalendar} entity in the S/4HANA system.
     * 
     * @param cndnContrSettlementCalendar
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrSettlementCalendar
     *            CndnContrSettlementCalendar} entity object that will be deleted in the S/4HANA system.
     * @return A request builder to delete an existing
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrSettlementCalendar
     *         CndnContrSettlementCalendar} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.DeleteRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrSettlementCalendar>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    DeleteRequestBuilder<CndnContrSettlementCalendar>
        deleteCndnContrSettlementCalendar( @Nonnull final CndnContrSettlementCalendar cndnContrSettlementCalendar );

    /**
     * Fetch multiple {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContract
     * ConditionContract} entities.
     * 
     * @return A request builder to fetch multiple
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContract
     *         ConditionContract} entities. This request builder allows methods which modify the underlying query to be
     *         called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContract>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetAllRequestBuilder<ConditionContract> getAllConditionContract();

    /**
     * Fetch the number of entries from the
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContract
     * ConditionContract} entity collection matching the filter and search expressions.
     * 
     * @return A request builder to fetch the count of
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContract
     *         ConditionContract} entities. This request builder allows methods which modify the underlying query to be
     *         called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContract>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CountRequestBuilder<ConditionContract> countConditionContract();

    /**
     * Fetch a single {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContract
     * ConditionContract} entity using key fields.
     * 
     * @param conditionContract
     *            Condition Record
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @return A request builder to fetch a single
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContract
     *         ConditionContract} entity using key fields. This request builder allows methods which modify the
     *         underlying query to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContract>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetByKeyRequestBuilder<ConditionContract> getConditionContractByKey( @Nonnull final String conditionContract );

    /**
     * Create a new {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContract
     * ConditionContract} entity and save it to the S/4HANA system.
     * 
     * @param conditionContract
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContract
     *            ConditionContract} entity object that will be created in the S/4HANA system.
     * @return A request builder to create a new
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContract
     *         ConditionContract} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContract>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CreateRequestBuilder<ConditionContract>
        createConditionContract( @Nonnull final ConditionContract conditionContract );

    /**
     * Update an existing
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContract
     * ConditionContract} entity and save it to the S/4HANA system.
     * 
     * @param conditionContract
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContract
     *            ConditionContract} entity object that will be updated in the S/4HANA system.
     * @return A request builder to update an existing
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContract
     *         ConditionContract} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContract>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    UpdateRequestBuilder<ConditionContract>
        updateConditionContract( @Nonnull final ConditionContract conditionContract );

    /**
     * Fetch multiple
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContractStatus
     * ConditionContractStatus} entities.
     * 
     * @return A request builder to fetch multiple
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContractStatus
     *         ConditionContractStatus} entities. This request builder allows methods which modify the underlying query
     *         to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContractStatus>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetAllRequestBuilder<ConditionContractStatus> getAllConditionContractStatus();

    /**
     * Fetch the number of entries from the
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContractStatus
     * ConditionContractStatus} entity collection matching the filter and search expressions.
     * 
     * @return A request builder to fetch the count of
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContractStatus
     *         ConditionContractStatus} entities. This request builder allows methods which modify the underlying query
     *         to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContractStatus>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CountRequestBuilder<ConditionContractStatus> countConditionContractStatus();

    /**
     * Fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContractStatus
     * ConditionContractStatus} entity using key fields.
     * 
     * @param conditionContract
     *            Condition Contract
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param statusCode
     *            Object Status
     *            <p>
     *            Constraints: Not nullable, Maximum length: 5
     *            </p>
     * @return A request builder to fetch a single
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContractStatus
     *         ConditionContractStatus} entity using key fields. This request builder allows methods which modify the
     *         underlying query to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContractStatus>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetByKeyRequestBuilder<ConditionContractStatus>
        getConditionContractStatusByKey( @Nonnull final String conditionContract, @Nonnull final String statusCode );

    /**
     * Create a new
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContractStatus
     * ConditionContractStatus} entity and save it to the S/4HANA system.
     * 
     * @param conditionContractStatus
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContractStatus
     *            ConditionContractStatus} entity object that will be created in the S/4HANA system.
     * @return A request builder to create a new
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContractStatus
     *         ConditionContractStatus} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContractStatus>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CreateRequestBuilder<ConditionContractStatus>
        createConditionContractStatus( @Nonnull final ConditionContractStatus conditionContractStatus );

}
