/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.time.LocalDate;

import javax.annotation.Nonnull;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.DeleteRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemBusLock;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemLogical;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemPaytData;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.DocumentHeader;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.GLItem;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.GLItemPaytData;

/**
 * <p>
 * 
 * Retrieve a list of documents. Read data on document header, document business partner line items and document general
 * ledger line items.
 * 
 * Update document header and document business partner line items. Create and delete business locks on existing
 * business partner line items..
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/7f/7caf54278145009e3c0e9678d8ffa6/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/CADOCUMENTMANAGE_0001?section=OVERVIEW'>SAP API
 * Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>CADOCUMENTMANAGE_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Finance - Contract Accounting Open Item Management Integration (SAP_COM_0361)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/2UJ'>Contract Accounting - Open Item Management (2UJ)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_CADOCUMENTMANAGE</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>ContrAcctgDocument</td>
 * </tr>
 * </table>
 * 
 */
public interface ContractAccountingDocumentManageService
{

    /**
     * If no other path was provided via the {@link #withServicePath(String)} method, this is the default service path
     * used to access the endpoint.
     * 
     */
    String DEFAULT_SERVICE_PATH = "/sap/opu/odata4/sap/api_cadocumentmanage/srvd_a2x/sap/cadocumentmanage/0001";

    /**
     * Overrides the default service path and returns a new service instance with the specified service path. Also
     * adjusts the respective entity URLs.
     * 
     * @param servicePath
     *            Service path that will override the default.
     * @return A new service instance with the specified service path.
     */
    @Nonnull
    ContractAccountingDocumentManageService withServicePath( @Nonnull final String servicePath );

    /**
     * Creates a batch request builder object.
     * 
     * @return A request builder to handle batch operation on this service. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder#execute(HttpDestinationProperties)
     *         execute} method on the request builder object.
     */
    @Nonnull
    BatchRequestBuilder batch();

    /**
     * Fetch multiple
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemBusLock
     * BPItemBusLock} entities.
     * 
     * @return A request builder to fetch multiple
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemBusLock
     *         BPItemBusLock} entities. This request builder allows methods which modify the underlying query to be
     *         called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemBusLock>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetAllRequestBuilder<BPItemBusLock> getAllBPItemBusLock();

    /**
     * Fetch the number of entries from the
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemBusLock
     * BPItemBusLock} entity collection matching the filter and search expressions.
     * 
     * @return A request builder to fetch the count of
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemBusLock
     *         BPItemBusLock} entities. This request builder allows methods which modify the underlying query to be
     *         called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemBusLock>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CountRequestBuilder<BPItemBusLock> countBPItemBusLock();

    /**
     * Fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemBusLock
     * BPItemBusLock} entity using key fields.
     * 
     * @param cABusinessLockValidTo
     *            Lock valid to
     *            <p>
     *            Constraints: Not nullable
     *            </p>
     * @param cADocumentNumber
     *            <p>
     *            Constraints: Not nullable, Maximum length: 12
     *            </p>
     * @param cABPItemNumber
     *            Item Number in FI-CA Document
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param cARepetitionItemNumber
     *            Repetition Item in FI-CA Document
     *            <p>
     *            Constraints: Not nullable, Maximum length: 3
     *            </p>
     * @param cABusinessLockValidFrom
     *            Lock valid from
     *            <p>
     *            Constraints: Not nullable
     *            </p>
     * @param cABusinessLockReason
     *            Lock Reason
     *            <p>
     *            Constraints: Not nullable, Maximum length: 1
     *            </p>
     * @param cABusinessProcess
     *            Process Code (Example: Dunning, Payment)
     *            <p>
     *            Constraints: Not nullable, Maximum length: 2
     *            </p>
     * @param cASubItemNumber
     *            Subitem for a Partial Clearing in Document
     *            <p>
     *            Constraints: Not nullable, Maximum length: 3
     *            </p>
     * @return A request builder to fetch a single
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemBusLock
     *         BPItemBusLock} entity using key fields. This request builder allows methods which modify the underlying
     *         query to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemBusLock>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetByKeyRequestBuilder<BPItemBusLock> getBPItemBusLockByKey(
        @Nonnull final String cADocumentNumber,
        @Nonnull final String cARepetitionItemNumber,
        @Nonnull final String cABPItemNumber,
        @Nonnull final String cASubItemNumber,
        @Nonnull final String cABusinessProcess,
        @Nonnull final String cABusinessLockReason,
        @Nonnull final LocalDate cABusinessLockValidFrom,
        @Nonnull final LocalDate cABusinessLockValidTo );

    /**
     * Create a new
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemBusLock
     * BPItemBusLock} entity and save it to the S/4HANA system.
     * 
     * @param bPItemBusLock
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemBusLock
     *            BPItemBusLock} entity object that will be created in the S/4HANA system.
     * @return A request builder to create a new
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemBusLock
     *         BPItemBusLock} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemBusLock>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CreateRequestBuilder<BPItemBusLock> createBPItemBusLock( @Nonnull final BPItemBusLock bPItemBusLock );

    /**
     * Deletes an existing
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemBusLock
     * BPItemBusLock} entity in the S/4HANA system.
     * 
     * @param bPItemBusLock
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemBusLock
     *            BPItemBusLock} entity object that will be deleted in the S/4HANA system.
     * @return A request builder to delete an existing
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemBusLock
     *         BPItemBusLock} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.DeleteRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemBusLock>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    DeleteRequestBuilder<BPItemBusLock> deleteBPItemBusLock( @Nonnull final BPItemBusLock bPItemBusLock );

    /**
     * Fetch multiple
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemLogical
     * BPItemLogical} entities.
     * 
     * @return A request builder to fetch multiple
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemLogical
     *         BPItemLogical} entities. This request builder allows methods which modify the underlying query to be
     *         called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemLogical>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetAllRequestBuilder<BPItemLogical> getAllBPItemLogical();

    /**
     * Fetch the number of entries from the
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemLogical
     * BPItemLogical} entity collection matching the filter and search expressions.
     * 
     * @return A request builder to fetch the count of
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemLogical
     *         BPItemLogical} entities. This request builder allows methods which modify the underlying query to be
     *         called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemLogical>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CountRequestBuilder<BPItemLogical> countBPItemLogical();

    /**
     * Fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemLogical
     * BPItemLogical} entity using key fields.
     * 
     * @param cADocumentNumber
     *            Number of a FI-CA Document
     *            <p>
     *            Constraints: Not nullable, Maximum length: 12
     *            </p>
     * @param cABPItemNumber
     *            Item Number in FI-CA Document
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param cARepetitionItemNumber
     *            Repetition Item in FI-CA Document
     *            <p>
     *            Constraints: Not nullable, Maximum length: 3
     *            </p>
     * @param cASubItemNumber
     *            Subitem for a Partial Clearing in Document
     *            <p>
     *            Constraints: Not nullable, Maximum length: 3
     *            </p>
     * @return A request builder to fetch a single
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemLogical
     *         BPItemLogical} entity using key fields. This request builder allows methods which modify the underlying
     *         query to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemLogical>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetByKeyRequestBuilder<BPItemLogical> getBPItemLogicalByKey(
        @Nonnull final String cADocumentNumber,
        @Nonnull final String cARepetitionItemNumber,
        @Nonnull final String cABPItemNumber,
        @Nonnull final String cASubItemNumber );

    /**
     * Create a new
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemLogical
     * BPItemLogical} entity and save it to the S/4HANA system.
     * 
     * @param bPItemLogical
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemLogical
     *            BPItemLogical} entity object that will be created in the S/4HANA system.
     * @return A request builder to create a new
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemLogical
     *         BPItemLogical} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemLogical>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CreateRequestBuilder<BPItemLogical> createBPItemLogical( @Nonnull final BPItemLogical bPItemLogical );

    /**
     * Update an existing
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemLogical
     * BPItemLogical} entity and save it to the S/4HANA system.
     * 
     * @param bPItemLogical
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemLogical
     *            BPItemLogical} entity object that will be updated in the S/4HANA system.
     * @return A request builder to update an existing
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemLogical
     *         BPItemLogical} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemLogical>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    UpdateRequestBuilder<BPItemLogical> updateBPItemLogical( @Nonnull final BPItemLogical bPItemLogical );

    /**
     * Fetch multiple
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemPaytData
     * BPItemPaytData} entities.
     * 
     * @return A request builder to fetch multiple
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemPaytData
     *         BPItemPaytData} entities. This request builder allows methods which modify the underlying query to be
     *         called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemPaytData>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetAllRequestBuilder<BPItemPaytData> getAllBPItemPaytData();

    /**
     * Fetch the number of entries from the
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemPaytData
     * BPItemPaytData} entity collection matching the filter and search expressions.
     * 
     * @return A request builder to fetch the count of
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemPaytData
     *         BPItemPaytData} entities. This request builder allows methods which modify the underlying query to be
     *         called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemPaytData>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CountRequestBuilder<BPItemPaytData> countBPItemPaytData();

    /**
     * Fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemPaytData
     * BPItemPaytData} entity using key fields.
     * 
     * @param cADocumentNumber
     *            Number of a FI-CA Document
     *            <p>
     *            Constraints: Not nullable, Maximum length: 12
     *            </p>
     * @param cAGroupingForPayment
     *            Grouping field for automatic payments
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @return A request builder to fetch a single
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemPaytData
     *         BPItemPaytData} entity using key fields. This request builder allows methods which modify the underlying
     *         query to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemPaytData>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetByKeyRequestBuilder<BPItemPaytData>
        getBPItemPaytDataByKey( @Nonnull final String cADocumentNumber, @Nonnull final String cAGroupingForPayment );

    /**
     * Fetch multiple
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.DocumentHeader
     * DocumentHeader} entities.
     * 
     * @return A request builder to fetch multiple
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.DocumentHeader
     *         DocumentHeader} entities. This request builder allows methods which modify the underlying query to be
     *         called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.DocumentHeader>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetAllRequestBuilder<DocumentHeader> getAllDocumentHeader();

    /**
     * Fetch the number of entries from the
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.DocumentHeader
     * DocumentHeader} entity collection matching the filter and search expressions.
     * 
     * @return A request builder to fetch the count of
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.DocumentHeader
     *         DocumentHeader} entities. This request builder allows methods which modify the underlying query to be
     *         called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.DocumentHeader>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CountRequestBuilder<DocumentHeader> countDocumentHeader();

    /**
     * Fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.DocumentHeader
     * DocumentHeader} entity using key fields.
     * 
     * @param cADocumentNumber
     *            Document
     *            <p>
     *            Constraints: Not nullable, Maximum length: 12
     *            </p>
     * @return A request builder to fetch a single
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.DocumentHeader
     *         DocumentHeader} entity using key fields. This request builder allows methods which modify the underlying
     *         query to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.DocumentHeader>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetByKeyRequestBuilder<DocumentHeader> getDocumentHeaderByKey( @Nonnull final String cADocumentNumber );

    /**
     * Update an existing
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.DocumentHeader
     * DocumentHeader} entity and save it to the S/4HANA system.
     * 
     * @param documentHeader
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.DocumentHeader
     *            DocumentHeader} entity object that will be updated in the S/4HANA system.
     * @return A request builder to update an existing
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.DocumentHeader
     *         DocumentHeader} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.DocumentHeader>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    UpdateRequestBuilder<DocumentHeader> updateDocumentHeader( @Nonnull final DocumentHeader documentHeader );

    /**
     * Fetch multiple
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.GLItem GLItem}
     * entities.
     * 
     * @return A request builder to fetch multiple
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.GLItem
     *         GLItem} entities. This request builder allows methods which modify the underlying query to be called
     *         before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.GLItem>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetAllRequestBuilder<GLItem> getAllGLItem();

    /**
     * Fetch the number of entries from the
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.GLItem GLItem}
     * entity collection matching the filter and search expressions.
     * 
     * @return A request builder to fetch the count of
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.GLItem
     *         GLItem} entities. This request builder allows methods which modify the underlying query to be called
     *         before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.GLItem>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CountRequestBuilder<GLItem> countGLItem();

    /**
     * Fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.GLItem GLItem}
     * entity using key fields.
     * 
     * @param cADocumentNumber
     *            Number of a FI-CA Document
     *            <p>
     *            Constraints: Not nullable, Maximum length: 12
     *            </p>
     * @param cAGLItemNumber
     *            Item Number in FI-CA Document
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @return A request builder to fetch a single
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.GLItem
     *         GLItem} entity using key fields. This request builder allows methods which modify the underlying query to
     *         be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.GLItem>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetByKeyRequestBuilder<GLItem>
        getGLItemByKey( @Nonnull final String cADocumentNumber, @Nonnull final String cAGLItemNumber );

    /**
     * Fetch multiple
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.GLItemPaytData
     * GLItemPaytData} entities.
     * 
     * @return A request builder to fetch multiple
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.GLItemPaytData
     *         GLItemPaytData} entities. This request builder allows methods which modify the underlying query to be
     *         called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.GLItemPaytData>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetAllRequestBuilder<GLItemPaytData> getAllGLItemPaytData();

    /**
     * Fetch the number of entries from the
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.GLItemPaytData
     * GLItemPaytData} entity collection matching the filter and search expressions.
     * 
     * @return A request builder to fetch the count of
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.GLItemPaytData
     *         GLItemPaytData} entities. This request builder allows methods which modify the underlying query to be
     *         called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.GLItemPaytData>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CountRequestBuilder<GLItemPaytData> countGLItemPaytData();

    /**
     * Fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.GLItemPaytData
     * GLItemPaytData} entity using key fields.
     * 
     * @param cADocumentNumber
     *            Number of a FI-CA Document
     *            <p>
     *            Constraints: Not nullable, Maximum length: 12
     *            </p>
     * @param cAGLItemNumber
     *            Item Number in FI-CA Document
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @return A request builder to fetch a single
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.GLItemPaytData
     *         GLItemPaytData} entity using key fields. This request builder allows methods which modify the underlying
     *         query to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.GLItemPaytData>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetByKeyRequestBuilder<GLItemPaytData>
        getGLItemPaytDataByKey( @Nonnull final String cADocumentNumber, @Nonnull final String cAGLItemNumber );

}
