/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.accesssequenceforpricinginsales.SalesPricingAccess;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.accesssequenceforpricinginsales.SalesPricingAccessField;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.accesssequenceforpricinginsales.SalesPricingAccessSequence;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.accesssequenceforpricinginsales.SlsPricingAccessSequenceText;

import lombok.Getter;

/**
 * <p>
 * Consumer of this inbound service can read the access sequences inside your system by sending OData requests.
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com/viewer/03c04db2a7434731b7fe21dca77440da/latest/en-US/df34f53fa1bc4d0ca0bb7dd8b25c7aec.html'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/SLSPRCGACCESSSEQUENCE_0001?section=OVERVIEW'>SAP
 * API Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>SLSPRCGACCESSSEQUENCE_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Pricing Data Integration (SAP_COM_0294)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/16T'>Intercompany Processes - Project-Based Services (16T)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/18J'>Requisitioning (18J)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/19C'>Activity Management in Procurement (19C)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/19E'>Supplier Classification and Segmentation (19E)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/1A0'>Sourcing with SAP Ariba Sourcing (1A0)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/1A8'>Internal Project Management - Project-Based Services (1A8)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/1B6'>Sales Rebate Processing (1B6)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/1EZ'>Credit Memo Processing (1EZ)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/1F1'>Debit Memo Processing (1F1)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/1HO'>Intercompany Sales Order Processing - Domestic (1HO)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/1IQ'>Sales Inquiry (1IQ)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/1IU'>Customer Consignment (1IU)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/1JW'>Advanced Available-to-Promise Processing (1JW)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/1KC'>Advanced Resource Management - Project-Based Services (1KC)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/1LE'>Invoice Processing by OpenText (1LE)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/1MC'>Convergent Billing (1MC)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/1MI'>Delivery Processing without Order Reference (1MI)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/1MX'>Intercompany Sales Order Processing - International (1MX)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/1NB'>Ariba - Sales Order to Invoice integration for SD (1NB)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/1WO'>Extended Intercompany Processes for Sales (1WO)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/1XF'>Request for Price (1XF)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/1Z1'>Digital Payments - Sales (1Z1)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/1Z3'>Accelerated Third-Party Returns (1Z3)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/1Z6'>External Billing (1Z6)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/22Z'>Procurement of Services (22Z)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/2EJ'>Integration of Procurement with External Suppliers (2EJ)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/2EL'>Integration of Sales with External Buyers (2EL)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/2EQ'>Sale of Services (2EQ)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/2ET'>Sales Order Processing for Non-Stock Material (2ET)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/2F4'>Intercompany Process for Customer Returns (2F4)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/2LG'>Supplier Consignment (2LG)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/2LH'>Automated Invoice Settlement (2LH)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/2LN'>Basic Available-to-Promise Processing (2LN)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/2ME'>Central Purchase Contracts (2ME)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/2MV'>Basic Resource Management - Project-Based Services (2MV)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/2NV'>Guided Buying Capability with SAP Ariba Buying (2NV)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/2NX'>Scheduling Agreement for Subcontracting Procurement (2NX)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/2NY'>Scheduling Agreement for Consignment Procurement (2NY)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/2R3'>Cross-Company Purchasing Organization (2R3)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/2R7'>Purchasing Rebate Processing (2R7)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/2TT'>Sales Commissions - External Sales Representative (2TT)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/2TW'>Complementary Procurement Functions (2TW)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/2XU'>Procurement of Materials with Variant Configuration (2XU)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/2XV'>Propose Material Group in Free-Text Purchase Requisitions
 * (2XV)</a>, <a href='https://rapid.sap.com/bp/#/scopeitems/2XW'>Propose Creation of Catalog Items based on Free-Text
 * Purchase Orders (2XW)</a>, <a href='https://rapid.sap.com/bp/#/scopeitems/31Q'>Sales Order Processing for Sales Kits
 * (31Q)</a>, <a href='https://rapid.sap.com/bp/#/scopeitems/3D2'>Service Order Management and Monitoring (3D2)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BD3'>Sales Processing using Third-Party with Shipping Notification
 * (BD3)</a>, <a href='https://rapid.sap.com/bp/#/scopeitems/BD9'>Sell from Stock (BD9)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BDA'>Free of Charge Delivery (BDA)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BDD'>Customer Returns (BDD)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BDG'>Sales Quotation (BDG)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BDH'>Sales Order Entry with One-Time Customer (BDH)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BDK'>Sales Processing using Third-Party without Shipping Notification
 * (BDK)</a>, <a href='https://rapid.sap.com/bp/#/scopeitems/BDN'>Sales of Non-Stock Item with Order-Specific
 * Procurement (BDN)</a>, <a href='https://rapid.sap.com/bp/#/scopeitems/BDQ'>Invoice Correction Process with Debit Memo
 * (BDQ)</a>, <a href='https://rapid.sap.com/bp/#/scopeitems/BDW'>Returnables Processing (BDW)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BKA'>Free Goods Processing (BKA)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BKJ'>Sales Order Processing with Customer Down Payment (BKJ)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BKK'>Sales Order Fulfillment Monitoring (BKK)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BKL'>Invoice Correction Process with Credit Memo (BKL)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BKN'>Order-to-Cash Performance Monitoring (BKN)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BKP'>Accelerated Customer Returns (BKP)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BKX'>Sales Order Processing - SEPA Direct Debit Handling (BKX)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BKZ'>Sales Order Processing with Invoice List and Collective Billing
 * (BKZ)</a>, <a href='https://rapid.sap.com/bp/#/scopeitems/BMD'>Purchase Contract (BMD)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BMK'>Return to Supplier (BMK)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BMR'>Scheduling Agreements in Procurement (BMR)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BMY'>Subcontracting (BMY)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BNX'>Consumable Purchasing (BNX)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/I9I'>Sales Contract Management (I9I)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/J11'>Customer Project Management - Project-Based Services (J11)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/J12'>Time Recording - Project-Based Services (J12)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/J13'>Service and Material Procurement - Project-Based Services
 * (J13)</a>, <a href='https://rapid.sap.com/bp/#/scopeitems/J14'>Sales Order Processing - Project-Based Services
 * (J14)</a>, <a href='https://rapid.sap.com/bp/#/scopeitems/J45'>Procurement of Direct Materials (J45)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/J82'>Automated Purchase-to-Pay with SAP Ariba Commerce Automation
 * (J82)</a>, <a href='https://rapid.sap.com/bp/#/scopeitems/O45'>Procurement of Direct Materials (O45)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/SL4'>Supplier Evaluation and Performance Monitoring (SL4)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_SLSPRCGACCESSSEQUENCE</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>SalesPricingAccessSequence</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultAccessSequenceForPricingInSalesService" )
public class DefaultAccessSequenceForPricingInSalesService
    implements
    ServiceWithNavigableEntities,
    AccessSequenceForPricingInSalesService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link AccessSequenceForPricingInSalesService#DEFAULT_SERVICE_PATH} to send the requests.
     * 
     */
    public DefaultAccessSequenceForPricingInSalesService()
    {
        servicePath = AccessSequenceForPricingInSalesService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultAccessSequenceForPricingInSalesService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultAccessSequenceForPricingInSalesService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultAccessSequenceForPricingInSalesService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<SalesPricingAccess> getAllSalesPricingAccess()
    {
        return new GetAllRequestBuilder<SalesPricingAccess>(
            servicePath,
            SalesPricingAccess.class,
            "SalesPricingAccess");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<SalesPricingAccess> countSalesPricingAccess()
    {
        return new CountRequestBuilder<SalesPricingAccess>(servicePath, SalesPricingAccess.class, "SalesPricingAccess");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<SalesPricingAccess> getSalesPricingAccessByKey(
        @Nonnull final String conditionUsage,
        @Nonnull final String conditionApplication,
        @Nonnull final String accessSequence,
        @Nonnull final String accessNumberOfAccessSequence )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("ConditionUsage", conditionUsage);
        key.put("ConditionApplication", conditionApplication);
        key.put("AccessSequence", accessSequence);
        key.put("AccessNumberOfAccessSequence", accessNumberOfAccessSequence);
        return new GetByKeyRequestBuilder<SalesPricingAccess>(
            servicePath,
            SalesPricingAccess.class,
            key,
            "SalesPricingAccess");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<SalesPricingAccessField> getAllSalesPricingAccessField()
    {
        return new GetAllRequestBuilder<SalesPricingAccessField>(
            servicePath,
            SalesPricingAccessField.class,
            "SalesPricingAccessField");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<SalesPricingAccessField> countSalesPricingAccessField()
    {
        return new CountRequestBuilder<SalesPricingAccessField>(
            servicePath,
            SalesPricingAccessField.class,
            "SalesPricingAccessField");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<SalesPricingAccessField> getSalesPricingAccessFieldByKey(
        @Nonnull final String conditionUsage,
        @Nonnull final String conditionApplication,
        @Nonnull final String accessSequence,
        @Nonnull final String accessNumberOfAccessSequence,
        @Nonnull final String conditionTable,
        @Nonnull final String pricingConditionField )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("ConditionUsage", conditionUsage);
        key.put("ConditionApplication", conditionApplication);
        key.put("AccessSequence", accessSequence);
        key.put("AccessNumberOfAccessSequence", accessNumberOfAccessSequence);
        key.put("ConditionTable", conditionTable);
        key.put("PricingConditionField", pricingConditionField);
        return new GetByKeyRequestBuilder<SalesPricingAccessField>(
            servicePath,
            SalesPricingAccessField.class,
            key,
            "SalesPricingAccessField");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<SalesPricingAccessSequence> getAllSalesPricingAccessSequence()
    {
        return new GetAllRequestBuilder<SalesPricingAccessSequence>(
            servicePath,
            SalesPricingAccessSequence.class,
            "SalesPricingAccessSequence");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<SalesPricingAccessSequence> countSalesPricingAccessSequence()
    {
        return new CountRequestBuilder<SalesPricingAccessSequence>(
            servicePath,
            SalesPricingAccessSequence.class,
            "SalesPricingAccessSequence");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<SalesPricingAccessSequence> getSalesPricingAccessSequenceByKey(
        @Nonnull final String conditionUsage,
        @Nonnull final String conditionApplication,
        @Nonnull final String accessSequence )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("ConditionUsage", conditionUsage);
        key.put("ConditionApplication", conditionApplication);
        key.put("AccessSequence", accessSequence);
        return new GetByKeyRequestBuilder<SalesPricingAccessSequence>(
            servicePath,
            SalesPricingAccessSequence.class,
            key,
            "SalesPricingAccessSequence");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<SlsPricingAccessSequenceText> getAllSlsPricingAccessSequenceText()
    {
        return new GetAllRequestBuilder<SlsPricingAccessSequenceText>(
            servicePath,
            SlsPricingAccessSequenceText.class,
            "SlsPricingAccessSequenceText");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<SlsPricingAccessSequenceText> countSlsPricingAccessSequenceText()
    {
        return new CountRequestBuilder<SlsPricingAccessSequenceText>(
            servicePath,
            SlsPricingAccessSequenceText.class,
            "SlsPricingAccessSequenceText");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<SlsPricingAccessSequenceText> getSlsPricingAccessSequenceTextByKey(
        @Nonnull final String language,
        @Nonnull final String conditionUsage,
        @Nonnull final String conditionApplication,
        @Nonnull final String accessSequence )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("Language", language);
        key.put("ConditionUsage", conditionUsage);
        key.put("ConditionApplication", conditionApplication);
        key.put("AccessSequence", accessSequence);
        return new GetByKeyRequestBuilder<SlsPricingAccessSequenceText>(
            servicePath,
            SlsPricingAccessSequenceText.class,
            key,
            "SlsPricingAccessSequenceText");
    }

}
