/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.bankv2.Bank;

import lombok.Getter;

/**
 * <p>
 * 
 * You can use this inbound synchronous service to create, read, update, and mark bank master data for deletion. Use a
 * standard archiving tool for the actual deletion of bank master data. The service is based on the OData protocol and
 * can be consumed in SAP Fiori apps and other user interfaces. The service contains Bank Country, Bank Internal ID,
 * Bank Name, Region, Street Name, City Name, SWIFT Code, Bank Network Grouping, Is Marked For Deletion, Bank Branch and
 * Bank Category..
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/55/12cf2c6e2e45c8a141581ddd708647/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference:
 * <a href='https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/BANK_0002?section=OVERVIEW'>SAP API
 * Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>BANK_0002</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>2</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Bank - Bank Integration (SAP_COM_0379)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/BFA'>Basic Bank Account Management (BFA)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509, OAuth2</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_BANK</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>Bank</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultBankV2Service" )
public class DefaultBankV2Service implements ServiceWithNavigableEntities, BankV2Service
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link BankV2Service#DEFAULT_SERVICE_PATH} to send the requests.
     * 
     */
    public DefaultBankV2Service()
    {
        servicePath = BankV2Service.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultBankV2Service( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultBankV2Service withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultBankV2Service(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<Bank> getAllBank()
    {
        return new GetAllRequestBuilder<Bank>(servicePath, Bank.class, "Bank");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<Bank> countBank()
    {
        return new CountRequestBuilder<Bank>(servicePath, Bank.class, "Bank");
    }

    @Override
    @Nonnull
    public
        GetByKeyRequestBuilder<Bank>
        getBankByKey( @Nonnull final String bankCountry, @Nonnull final String bankInternalID )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("BankCountry", bankCountry);
        key.put("BankInternalID", bankInternalID);
        return new GetByKeyRequestBuilder<Bank>(servicePath, Bank.class, key, "Bank");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<Bank> createBank( @Nonnull final Bank bank )
    {
        return new CreateRequestBuilder<Bank>(servicePath, bank, "Bank");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<Bank> updateBank( @Nonnull final Bank bank )
    {
        return new UpdateRequestBuilder<Bank>(servicePath, bank, "Bank");
    }

}
