/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.cashsecuritydeposit.CAScrtyDepDocumentFlow;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.cashsecuritydeposit.CASecurityDeposit;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.cashsecuritydeposit.CASecurityDepositReqDetail;

import lombok.Getter;

/**
 * <p>
 * 
 * The service contains a list of security deposits. With this service you can create, read, update cash security
 * deposits. You can change header information, add request documents, reverse, clear, release or reset the release of
 * specific cash security deposits.
 * 
 * After a request to fetch security deposits is sent, all filtered security deposits are returned.
 * 
 * After a security deposit has been created using the Create operation a new security deposit number and the message
 * header are returned.
 * 
 * After a security deposit has been revised, for example, by using the Change or Create Request Operation operation,
 * the number of that revised security deposit and the message header are returned..
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/9c/9d40cd785b4317a85fd3d8e6c3a26e/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/CASECURITYDEPOSIT_0001?section=OVERVIEW'>SAP
 * API Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>CASECURITYDEPOSIT_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Finance - Contract Accounting Security Deposits Integration (SAP_COM_0380)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/2UJ'>Contract Accounting - Open Item Management (2UJ)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_CASECURITYDEPOSIT</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>ContrAcctgSecurityDeposit</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultCashSecurityDepositService" )
public class DefaultCashSecurityDepositService implements ServiceWithNavigableEntities, CashSecurityDepositService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link CashSecurityDepositService#DEFAULT_SERVICE_PATH} to send the requests.
     * 
     */
    public DefaultCashSecurityDepositService()
    {
        servicePath = CashSecurityDepositService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultCashSecurityDepositService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultCashSecurityDepositService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultCashSecurityDepositService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<CAScrtyDepDocumentFlow> getAllCAScrtyDepDocumentFlow()
    {
        return new GetAllRequestBuilder<CAScrtyDepDocumentFlow>(
            servicePath,
            CAScrtyDepDocumentFlow.class,
            "CAScrtyDepDocumentFlow");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<CAScrtyDepDocumentFlow> countCAScrtyDepDocumentFlow()
    {
        return new CountRequestBuilder<CAScrtyDepDocumentFlow>(
            servicePath,
            CAScrtyDepDocumentFlow.class,
            "CAScrtyDepDocumentFlow");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<CAScrtyDepDocumentFlow> getCAScrtyDepDocumentFlowByKey(
        @Nonnull final String cASecurityDeposit,
        @Nonnull final String cADocumentNumber,
        @Nonnull final String cAScrtyDepositClearedDocument )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("CASecurityDeposit", cASecurityDeposit);
        key.put("CADocumentNumber", cADocumentNumber);
        key.put("CAScrtyDepositClearedDocument", cAScrtyDepositClearedDocument);
        return new GetByKeyRequestBuilder<CAScrtyDepDocumentFlow>(
            servicePath,
            CAScrtyDepDocumentFlow.class,
            key,
            "CAScrtyDepDocumentFlow");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<CASecurityDeposit> getAllCASecurityDeposit()
    {
        return new GetAllRequestBuilder<CASecurityDeposit>(servicePath, CASecurityDeposit.class, "CASecurityDeposit");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<CASecurityDeposit> countCASecurityDeposit()
    {
        return new CountRequestBuilder<CASecurityDeposit>(servicePath, CASecurityDeposit.class, "CASecurityDeposit");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<CASecurityDeposit> getCASecurityDepositByKey(
        @Nonnull final String cASecurityDeposit )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("CASecurityDeposit", cASecurityDeposit);
        return new GetByKeyRequestBuilder<CASecurityDeposit>(
            servicePath,
            CASecurityDeposit.class,
            key,
            "CASecurityDeposit");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<CASecurityDeposit> createCASecurityDeposit(
        @Nonnull final CASecurityDeposit cASecurityDeposit )
    {
        return new CreateRequestBuilder<CASecurityDeposit>(servicePath, cASecurityDeposit, "CASecurityDeposit");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<CASecurityDepositReqDetail> getAllCASecurityDepositReqDetail()
    {
        return new GetAllRequestBuilder<CASecurityDepositReqDetail>(
            servicePath,
            CASecurityDepositReqDetail.class,
            "CASecurityDepositReqDetail");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<CASecurityDepositReqDetail> countCASecurityDepositReqDetail()
    {
        return new CountRequestBuilder<CASecurityDepositReqDetail>(
            servicePath,
            CASecurityDepositReqDetail.class,
            "CASecurityDepositReqDetail");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<CASecurityDepositReqDetail> getCASecurityDepositReqDetailByKey(
        @Nonnull final String cASecurityDeposit,
        @Nonnull final String cADocumentNumber )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("CASecurityDeposit", cASecurityDeposit);
        key.put("CADocumentNumber", cADocumentNumber);
        return new GetByKeyRequestBuilder<CASecurityDepositReqDetail>(
            servicePath,
            CASecurityDepositReqDetail.class,
            key,
            "CASecurityDepositReqDetail");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<CASecurityDepositReqDetail> createCASecurityDepositReqDetail(
        @Nonnull final CASecurityDepositReqDetail cASecurityDepositReqDetail )
    {
        return new CreateRequestBuilder<CASecurityDepositReqDetail>(
            servicePath,
            cASecurityDepositReqDetail,
            "CASecurityDepositReqDetail");
    }

}
