/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.catalogtypecoding.MaintNotifCodingCode;

import lombok.Getter;

/**
 * <p>
 * 
 * This service contains catalog groups and codes that belong to catalog type D which is Coding. The catalog type Coding
 * is used in maintenance notification to record the failure modes. The information is sent in the request as a payload.
 * 
 * The request payload contains all information required to read the catalog type Coding. All the information in the
 * payload is used to read Coding as a catalog type and includes appropriate checks for restricted values and
 * authorizations. Once the coding has been read, the response with the data is included.
 * 
 * If the input in the request is not as expected when you read the catalog type Coding, the system displays appropriate
 * error or warning messages in the response. .
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/b0/ecb19d4b3f459f84644a54eb28d693/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference:
 * <a href='https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/CODING_0001?section=OVERVIEW'>SAP API
 * Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>CODING_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Maintenance Processing Integration (SAP_COM_0397)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/4HH'>Reactive Maintenance (4HH)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/4HI'>Proactive Maintenance (4HI)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/4VT'>Improvement Maintenance (4VT)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/4WM'>Operational and Overhead Maintenance (4WM)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BH1'>Corrective Maintenance (BH1)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BH2'>Emergency Maintenance (BH2)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BJ2'>Preventive Maintenance (BJ2)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509, OAuth2</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_CODING</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>MaintenanceNotification</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultCatalogTypeCodingService" )
public class DefaultCatalogTypeCodingService implements ServiceWithNavigableEntities, CatalogTypeCodingService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link CatalogTypeCodingService#DEFAULT_SERVICE_PATH} to send the requests.
     * 
     */
    public DefaultCatalogTypeCodingService()
    {
        servicePath = CatalogTypeCodingService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultCatalogTypeCodingService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultCatalogTypeCodingService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultCatalogTypeCodingService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<MaintNotifCodingCode> getAllMaintNotifCodingCode()
    {
        return new GetAllRequestBuilder<MaintNotifCodingCode>(
            servicePath,
            MaintNotifCodingCode.class,
            "MaintNotifCodingCode");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<MaintNotifCodingCode> countMaintNotifCodingCode()
    {
        return new CountRequestBuilder<MaintNotifCodingCode>(
            servicePath,
            MaintNotifCodingCode.class,
            "MaintNotifCodingCode");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<MaintNotifCodingCode> getMaintNotifCodingCodeByKey(
        @Nonnull final String maintNotifCodingCatalog,
        @Nonnull final String maintNotifCodingCodeGroup,
        @Nonnull final String maintNotifCodingCode )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("MaintNotifCodingCatalog", maintNotifCodingCatalog);
        key.put("MaintNotifCodingCodeGroup", maintNotifCodingCodeGroup);
        key.put("MaintNotifCodingCode", maintNotifCodingCode);
        return new GetByKeyRequestBuilder<MaintNotifCodingCode>(
            servicePath,
            MaintNotifCodingCode.class,
            key,
            "MaintNotifCodingCode");
    }

}
