/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.catalogtypeobjectpart.MaintNotifObjPrtCode;

import lombok.Getter;

/**
 * <p>
 * 
 * This service contains catalog groups and codes that belong to catalog type B which is Object Parts. The catalog type
 * Object Parts is used in maintenance notification item level to record the object parts. The information is sent in
 * the request as a payload.
 * 
 * The request payload contains all information required to read the catalog type Object Parts. All the information in
 * the payload is used to read Object Parts as a catalog type and includes appropriate checks for restricted values and
 * authorizations. Once the object parts have been read, the response with the data is included.
 * 
 * If the input in the request is not as expected when you read the catalog type Object Parts, the system displays
 * appropriate error or warning messages in the response..
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/ed/6fe185ade647078c9a0c00b576e047/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference:
 * <a href='https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/OBJECTPART_0001?section=OVERVIEW'>SAP
 * API Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>OBJECTPART_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Maintenance Processing Integration (SAP_COM_0397)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/4HH'>Reactive Maintenance (4HH)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/4HI'>Proactive Maintenance (4HI)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/4VT'>Improvement Maintenance (4VT)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/4WM'>Operational and Overhead Maintenance (4WM)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BH1'>Corrective Maintenance (BH1)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BH2'>Emergency Maintenance (BH2)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BJ2'>Preventive Maintenance (BJ2)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509, OAuth2</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_OBJECTPART</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>MaintenanceNotification</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultCatalogTypeObjectPartService" )
public class DefaultCatalogTypeObjectPartService implements ServiceWithNavigableEntities, CatalogTypeObjectPartService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link CatalogTypeObjectPartService#DEFAULT_SERVICE_PATH} to send the requests.
     * 
     */
    public DefaultCatalogTypeObjectPartService()
    {
        servicePath = CatalogTypeObjectPartService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultCatalogTypeObjectPartService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultCatalogTypeObjectPartService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultCatalogTypeObjectPartService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<MaintNotifObjPrtCode> getAllMaintNotifObjPrtCode()
    {
        return new GetAllRequestBuilder<MaintNotifObjPrtCode>(
            servicePath,
            MaintNotifObjPrtCode.class,
            "MaintNotifObjPrtCode");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<MaintNotifObjPrtCode> countMaintNotifObjPrtCode()
    {
        return new CountRequestBuilder<MaintNotifObjPrtCode>(
            servicePath,
            MaintNotifObjPrtCode.class,
            "MaintNotifObjPrtCode");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<MaintNotifObjPrtCode> getMaintNotifObjPrtCodeByKey(
        @Nonnull final String maintNotifObjPartCatalog,
        @Nonnull final String maintNotifObjPartCodeGroup,
        @Nonnull final String maintNotifObjPartCode )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("MaintNotifObjPartCatalog", maintNotifObjPartCatalog);
        key.put("MaintNotifObjPartCodeGroup", maintNotifObjPartCodeGroup);
        key.put("MaintNotifObjPartCode", maintNotifObjPartCode);
        return new GetByKeyRequestBuilder<MaintNotifObjPrtCode>(
            servicePath,
            MaintNotifObjPrtCode.class,
            key,
            "MaintNotifObjPrtCode");
    }

}
