/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.catalogtypeoverviewofdamage.MaintNotifOvwDamageCode;

import lombok.Getter;

/**
 * <p>
 * 
 * This service contains catalog groups and codes that belong to catalog type C which is Overview of Damage. The catalog
 * type Overview of Damage is used in maintenance notification item level to record the damages in the asset. The
 * information is sent in the request as a payload.
 * 
 * The request payload contains all information required to read the catalog type Overview of Damage. All the
 * information in the payload is used to read Overview of Damage as a catalog type and includes appropriate checks for
 * restricted values and authorizations. Once the overview of damage has been read, the response with the data is
 * included.
 * 
 * If the input in the request is not as expected when you read the catalog type Overview of Damage, the system displays
 * appropriate error or warning messages in the response..
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/7c/4881781dd143148a42035325e5172c/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference:
 * <a href='https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/OVWDAMAGE_0001?section=OVERVIEW'>SAP
 * API Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>OVWDAMAGE_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Maintenance Processing Integration (SAP_COM_0397)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/4HH'>Reactive Maintenance (4HH)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/4HI'>Proactive Maintenance (4HI)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/4VT'>Improvement Maintenance (4VT)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/4WM'>Operational and Overhead Maintenance (4WM)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BH1'>Corrective Maintenance (BH1)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BH2'>Emergency Maintenance (BH2)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BJ2'>Preventive Maintenance (BJ2)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509, OAuth2</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_OVWDAMAGE</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>MaintenanceNotification</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultCatalogTypeOverviewOfDamageService" )
public class DefaultCatalogTypeOverviewOfDamageService
    implements
    ServiceWithNavigableEntities,
    CatalogTypeOverviewOfDamageService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link CatalogTypeOverviewOfDamageService#DEFAULT_SERVICE_PATH} to send the requests.
     * 
     */
    public DefaultCatalogTypeOverviewOfDamageService()
    {
        servicePath = CatalogTypeOverviewOfDamageService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultCatalogTypeOverviewOfDamageService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultCatalogTypeOverviewOfDamageService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultCatalogTypeOverviewOfDamageService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<MaintNotifOvwDamageCode> getAllMaintNotifOvwDamageCode()
    {
        return new GetAllRequestBuilder<MaintNotifOvwDamageCode>(
            servicePath,
            MaintNotifOvwDamageCode.class,
            "MaintNotifOvwDamageCode");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<MaintNotifOvwDamageCode> countMaintNotifOvwDamageCode()
    {
        return new CountRequestBuilder<MaintNotifOvwDamageCode>(
            servicePath,
            MaintNotifOvwDamageCode.class,
            "MaintNotifOvwDamageCode");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<MaintNotifOvwDamageCode> getMaintNotifOvwDamageCodeByKey(
        @Nonnull final String maintNotifOvwDamageCatalog,
        @Nonnull final String maintNotifOvwDamageCodeGroup,
        @Nonnull final String maintNotifOvwDamageCode )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("MaintNotifOvwDamageCatalog", maintNotifOvwDamageCatalog);
        key.put("MaintNotifOvwDamageCodeGroup", maintNotifOvwDamageCodeGroup);
        key.put("MaintNotifOvwDamageCode", maintNotifOvwDamageCode);
        return new GetByKeyRequestBuilder<MaintNotifOvwDamageCode>(
            servicePath,
            MaintNotifOvwDamageCode.class,
            key,
            "MaintNotifOvwDamageCode");
    }

}
