/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.time.LocalDate;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.DeleteRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.BusVolSelectionCriteria;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrCndnRecordValidity;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecord;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrConditionRecordScale;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.CndnContrSettlementCalendar;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContract;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.conditioncontract.ConditionContractStatus;

import lombok.Getter;

/**
 * <p>
 * 
 * The service enables you to retrieve and process transactional data of Condition Contacts:
 * 
 * You can read, create and update Condition Contracts.
 * 
 * You can also release Condition Contracts, as well as lock and release them for settlement.
 * 
 * You can determine the accruals amounts for a specific Condition Contract and group them based on several aspects.
 * 
 * You can read and create new entries in the business volume selection criteria of the Condition Contract. Furthermore,
 * you can also update and delete the entries.
 * 
 * You can read and create new entries in the settlement calendar of the Condition Contract. Furthermore, you can also
 * update and delete the entries.
 * 
 * You can read the status of the Condition Contract.
 * 
 * You can read and create new condition records. Furthermore, you can update and delete them. It is also possible to
 * set a condition record as invalid and to revoke the invalidity.
 * 
 * You can read and change the validity of the condition records.
 * 
 * You can read and create new condition record scales in a condition record. It is also possible to delete them..
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/43/a9ffe7477f47c3958930b832ea52c5/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/CONDITIONCONTRACT_0001?section=OVERVIEW'>SAP
 * API Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>CONDITIONCONTRACT_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Logistics - Condition Contract Integration (SAP_COM_0729)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/1B6'>Sales Rebate Processing (1B6)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/2R7'>Purchasing Rebate Processing (2R7)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/2TT'>Sales Commissions - External Sales Representative (2TT)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/55I'>Royalties Settlement for Sales (55I)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_CONDITION_CONTRACT</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>ConditionContract</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultConditionContractService" )
public class DefaultConditionContractService implements ServiceWithNavigableEntities, ConditionContractService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link ConditionContractService#DEFAULT_SERVICE_PATH} to send the requests.
     * 
     */
    public DefaultConditionContractService()
    {
        servicePath = ConditionContractService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultConditionContractService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultConditionContractService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultConditionContractService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<BusVolSelectionCriteria> getAllBusVolSelectionCriteria()
    {
        return new GetAllRequestBuilder<BusVolSelectionCriteria>(
            servicePath,
            BusVolSelectionCriteria.class,
            "BusVolSelectionCriteria");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<BusVolSelectionCriteria> countBusVolSelectionCriteria()
    {
        return new CountRequestBuilder<BusVolSelectionCriteria>(
            servicePath,
            BusVolSelectionCriteria.class,
            "BusVolSelectionCriteria");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<BusVolSelectionCriteria> getBusVolSelectionCriteriaByKey(
        @Nonnull final String conditionContract,
        @Nonnull final UUID cndnContrBusVolUUID )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("ConditionContract", conditionContract);
        key.put("CndnContrBusVolUUID", cndnContrBusVolUUID);
        return new GetByKeyRequestBuilder<BusVolSelectionCriteria>(
            servicePath,
            BusVolSelectionCriteria.class,
            key,
            "BusVolSelectionCriteria");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<BusVolSelectionCriteria> createBusVolSelectionCriteria(
        @Nonnull final BusVolSelectionCriteria busVolSelectionCriteria )
    {
        return new CreateRequestBuilder<BusVolSelectionCriteria>(
            servicePath,
            busVolSelectionCriteria,
            "BusVolSelectionCriteria");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<BusVolSelectionCriteria> updateBusVolSelectionCriteria(
        @Nonnull final BusVolSelectionCriteria busVolSelectionCriteria )
    {
        return new UpdateRequestBuilder<BusVolSelectionCriteria>(
            servicePath,
            busVolSelectionCriteria,
            "BusVolSelectionCriteria");
    }

    @Override
    @Nonnull
    public DeleteRequestBuilder<BusVolSelectionCriteria> deleteBusVolSelectionCriteria(
        @Nonnull final BusVolSelectionCriteria busVolSelectionCriteria )
    {
        return new DeleteRequestBuilder<BusVolSelectionCriteria>(
            servicePath,
            busVolSelectionCriteria,
            "BusVolSelectionCriteria");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<CndnContrCndnRecordValidity> getAllCndnContrCndnRecordValidity()
    {
        return new GetAllRequestBuilder<CndnContrCndnRecordValidity>(
            servicePath,
            CndnContrCndnRecordValidity.class,
            "CndnContrCndnRecordValidity");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<CndnContrCndnRecordValidity> countCndnContrCndnRecordValidity()
    {
        return new CountRequestBuilder<CndnContrCndnRecordValidity>(
            servicePath,
            CndnContrCndnRecordValidity.class,
            "CndnContrCndnRecordValidity");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<CndnContrCndnRecordValidity> getCndnContrCndnRecordValidityByKey(
        @Nonnull final String conditionContract,
        @Nonnull final String conditionRecord,
        @Nonnull final LocalDate conditionValidityEndDate )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("ConditionContract", conditionContract);
        key.put("ConditionRecord", conditionRecord);
        key.put("ConditionValidityEndDate", conditionValidityEndDate);
        return new GetByKeyRequestBuilder<CndnContrCndnRecordValidity>(
            servicePath,
            CndnContrCndnRecordValidity.class,
            key,
            "CndnContrCndnRecordValidity");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<CndnContrCndnRecordValidity> createCndnContrCndnRecordValidity(
        @Nonnull final CndnContrCndnRecordValidity cndnContrCndnRecordValidity )
    {
        return new CreateRequestBuilder<CndnContrCndnRecordValidity>(
            servicePath,
            cndnContrCndnRecordValidity,
            "CndnContrCndnRecordValidity");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<CndnContrConditionRecord> getAllCndnContrConditionRecord()
    {
        return new GetAllRequestBuilder<CndnContrConditionRecord>(
            servicePath,
            CndnContrConditionRecord.class,
            "CndnContrConditionRecord");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<CndnContrConditionRecord> countCndnContrConditionRecord()
    {
        return new CountRequestBuilder<CndnContrConditionRecord>(
            servicePath,
            CndnContrConditionRecord.class,
            "CndnContrConditionRecord");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<CndnContrConditionRecord> getCndnContrConditionRecordByKey(
        @Nonnull final String conditionContract,
        @Nonnull final String conditionRecord )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("ConditionContract", conditionContract);
        key.put("ConditionRecord", conditionRecord);
        return new GetByKeyRequestBuilder<CndnContrConditionRecord>(
            servicePath,
            CndnContrConditionRecord.class,
            key,
            "CndnContrConditionRecord");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<CndnContrConditionRecord> createCndnContrConditionRecord(
        @Nonnull final CndnContrConditionRecord cndnContrConditionRecord )
    {
        return new CreateRequestBuilder<CndnContrConditionRecord>(
            servicePath,
            cndnContrConditionRecord,
            "CndnContrConditionRecord");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<CndnContrConditionRecord> updateCndnContrConditionRecord(
        @Nonnull final CndnContrConditionRecord cndnContrConditionRecord )
    {
        return new UpdateRequestBuilder<CndnContrConditionRecord>(
            servicePath,
            cndnContrConditionRecord,
            "CndnContrConditionRecord");
    }

    @Override
    @Nonnull
    public DeleteRequestBuilder<CndnContrConditionRecord> deleteCndnContrConditionRecord(
        @Nonnull final CndnContrConditionRecord cndnContrConditionRecord )
    {
        return new DeleteRequestBuilder<CndnContrConditionRecord>(
            servicePath,
            cndnContrConditionRecord,
            "CndnContrConditionRecord");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<CndnContrConditionRecordScale> getAllCndnContrConditionRecordScale()
    {
        return new GetAllRequestBuilder<CndnContrConditionRecordScale>(
            servicePath,
            CndnContrConditionRecordScale.class,
            "CndnContrConditionRecordScale");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<CndnContrConditionRecordScale> countCndnContrConditionRecordScale()
    {
        return new CountRequestBuilder<CndnContrConditionRecordScale>(
            servicePath,
            CndnContrConditionRecordScale.class,
            "CndnContrConditionRecordScale");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<CndnContrConditionRecordScale> getCndnContrConditionRecordScaleByKey(
        @Nonnull final String conditionContract,
        @Nonnull final String conditionRecord,
        @Nonnull final String conditionScaleLine )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("ConditionContract", conditionContract);
        key.put("ConditionRecord", conditionRecord);
        key.put("ConditionScaleLine", conditionScaleLine);
        return new GetByKeyRequestBuilder<CndnContrConditionRecordScale>(
            servicePath,
            CndnContrConditionRecordScale.class,
            key,
            "CndnContrConditionRecordScale");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<CndnContrConditionRecordScale> createCndnContrConditionRecordScale(
        @Nonnull final CndnContrConditionRecordScale cndnContrConditionRecordScale )
    {
        return new CreateRequestBuilder<CndnContrConditionRecordScale>(
            servicePath,
            cndnContrConditionRecordScale,
            "CndnContrConditionRecordScale");
    }

    @Override
    @Nonnull
    public DeleteRequestBuilder<CndnContrConditionRecordScale> deleteCndnContrConditionRecordScale(
        @Nonnull final CndnContrConditionRecordScale cndnContrConditionRecordScale )
    {
        return new DeleteRequestBuilder<CndnContrConditionRecordScale>(
            servicePath,
            cndnContrConditionRecordScale,
            "CndnContrConditionRecordScale");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<CndnContrSettlementCalendar> getAllCndnContrSettlementCalendar()
    {
        return new GetAllRequestBuilder<CndnContrSettlementCalendar>(
            servicePath,
            CndnContrSettlementCalendar.class,
            "CndnContrSettlementCalendar");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<CndnContrSettlementCalendar> countCndnContrSettlementCalendar()
    {
        return new CountRequestBuilder<CndnContrSettlementCalendar>(
            servicePath,
            CndnContrSettlementCalendar.class,
            "CndnContrSettlementCalendar");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<CndnContrSettlementCalendar> getCndnContrSettlementCalendarByKey(
        @Nonnull final String conditionContract,
        @Nonnull final LocalDate settlmtDate,
        @Nonnull final String settlmtDateCat )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("ConditionContract", conditionContract);
        key.put("SettlmtDate", settlmtDate);
        key.put("SettlmtDateCat", settlmtDateCat);
        return new GetByKeyRequestBuilder<CndnContrSettlementCalendar>(
            servicePath,
            CndnContrSettlementCalendar.class,
            key,
            "CndnContrSettlementCalendar");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<CndnContrSettlementCalendar> createCndnContrSettlementCalendar(
        @Nonnull final CndnContrSettlementCalendar cndnContrSettlementCalendar )
    {
        return new CreateRequestBuilder<CndnContrSettlementCalendar>(
            servicePath,
            cndnContrSettlementCalendar,
            "CndnContrSettlementCalendar");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<CndnContrSettlementCalendar> updateCndnContrSettlementCalendar(
        @Nonnull final CndnContrSettlementCalendar cndnContrSettlementCalendar )
    {
        return new UpdateRequestBuilder<CndnContrSettlementCalendar>(
            servicePath,
            cndnContrSettlementCalendar,
            "CndnContrSettlementCalendar");
    }

    @Override
    @Nonnull
    public DeleteRequestBuilder<CndnContrSettlementCalendar> deleteCndnContrSettlementCalendar(
        @Nonnull final CndnContrSettlementCalendar cndnContrSettlementCalendar )
    {
        return new DeleteRequestBuilder<CndnContrSettlementCalendar>(
            servicePath,
            cndnContrSettlementCalendar,
            "CndnContrSettlementCalendar");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<ConditionContract> getAllConditionContract()
    {
        return new GetAllRequestBuilder<ConditionContract>(servicePath, ConditionContract.class, "ConditionContract");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<ConditionContract> countConditionContract()
    {
        return new CountRequestBuilder<ConditionContract>(servicePath, ConditionContract.class, "ConditionContract");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<ConditionContract> getConditionContractByKey(
        @Nonnull final String conditionContract )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("ConditionContract", conditionContract);
        return new GetByKeyRequestBuilder<ConditionContract>(
            servicePath,
            ConditionContract.class,
            key,
            "ConditionContract");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<ConditionContract> createConditionContract(
        @Nonnull final ConditionContract conditionContract )
    {
        return new CreateRequestBuilder<ConditionContract>(servicePath, conditionContract, "ConditionContract");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<ConditionContract> updateConditionContract(
        @Nonnull final ConditionContract conditionContract )
    {
        return new UpdateRequestBuilder<ConditionContract>(servicePath, conditionContract, "ConditionContract");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<ConditionContractStatus> getAllConditionContractStatus()
    {
        return new GetAllRequestBuilder<ConditionContractStatus>(
            servicePath,
            ConditionContractStatus.class,
            "ConditionContractStatus");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<ConditionContractStatus> countConditionContractStatus()
    {
        return new CountRequestBuilder<ConditionContractStatus>(
            servicePath,
            ConditionContractStatus.class,
            "ConditionContractStatus");
    }

    @Override
    @Nonnull
    public
        GetByKeyRequestBuilder<ConditionContractStatus>
        getConditionContractStatusByKey( @Nonnull final String conditionContract, @Nonnull final String statusCode )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("ConditionContract", conditionContract);
        key.put("StatusCode", statusCode);
        return new GetByKeyRequestBuilder<ConditionContractStatus>(
            servicePath,
            ConditionContractStatus.class,
            key,
            "ConditionContractStatus");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<ConditionContractStatus> createConditionContractStatus(
        @Nonnull final ConditionContractStatus conditionContractStatus )
    {
        return new CreateRequestBuilder<ConditionContractStatus>(
            servicePath,
            conditionContractStatus,
            "ConditionContractStatus");
    }

}
