/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingbusinesspartnerinvoice.BusPartInvoiceCorrespnc;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingbusinesspartnerinvoice.BusPartInvoiceItemParameters;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingbusinesspartnerinvoice.BusPartInvoiceParameters;

import lombok.Getter;

/**
 * <p>
 * 
 * Read invoice items for a business partner and read the corresponding invoice headers with this service. For each
 * invoice you can read correspondence data including the printed document. .
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/6e/c0cd0d9806492695c7e3354fe54d24/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/API_CABUSPARTINVOICE_0001?section=OVERVIEW'>SAP
 * API Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>API_CABUSPARTINVOICE_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Finance - Contract Accounting Open Item Management Integration (SAP_COM_0361)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/2UJ'>Contract Accounting - Open Item Management (2UJ)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_CABUSPARTINVOICE</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>ContrAcctgDocContainer</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultContractAccountingBusinessPartnerInvoiceService" )
public class DefaultContractAccountingBusinessPartnerInvoiceService
    implements
    ServiceWithNavigableEntities,
    ContractAccountingBusinessPartnerInvoiceService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link ContractAccountingBusinessPartnerInvoiceService#DEFAULT_SERVICE_PATH} to send the
     * requests.
     * 
     */
    public DefaultContractAccountingBusinessPartnerInvoiceService()
    {
        servicePath = ContractAccountingBusinessPartnerInvoiceService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultContractAccountingBusinessPartnerInvoiceService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultContractAccountingBusinessPartnerInvoiceService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultContractAccountingBusinessPartnerInvoiceService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<BusPartInvoiceParameters> getBusPartInvoiceByKey(
        @Nonnull final String p_DisplayCurrency )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("P_DisplayCurrency", p_DisplayCurrency);
        return new GetByKeyRequestBuilder<BusPartInvoiceParameters>(
            servicePath,
            BusPartInvoiceParameters.class,
            key,
            "BusPartInvoice");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<BusPartInvoiceCorrespnc> getAllBusPartInvoiceCorrespnc()
    {
        return new GetAllRequestBuilder<BusPartInvoiceCorrespnc>(
            servicePath,
            BusPartInvoiceCorrespnc.class,
            "BusPartInvoiceCorrespnc");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<BusPartInvoiceCorrespnc> countBusPartInvoiceCorrespnc()
    {
        return new CountRequestBuilder<BusPartInvoiceCorrespnc>(
            servicePath,
            BusPartInvoiceCorrespnc.class,
            "BusPartInvoiceCorrespnc");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<BusPartInvoiceCorrespnc> getBusPartInvoiceCorrespncByKey(
        @Nonnull final UUID cADocumentContainerUUID,
        @Nonnull final String cACorrespondenceType,
        @Nonnull final String cACorrespondence,
        @Nonnull final String cADocCtnCorrespncSqncNumber )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("CADocumentContainerUUID", cADocumentContainerUUID);
        key.put("CACorrespondenceType", cACorrespondenceType);
        key.put("CACorrespondence", cACorrespondence);
        key.put("CADocCtnCorrespncSqncNumber", cADocCtnCorrespncSqncNumber);
        return new GetByKeyRequestBuilder<BusPartInvoiceCorrespnc>(
            servicePath,
            BusPartInvoiceCorrespnc.class,
            key,
            "BusPartInvoiceCorrespnc");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<BusPartInvoiceItemParameters> getBusPartInvoiceItemByKey(
        @Nonnull final String p_DisplayCurrency )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("P_DisplayCurrency", p_DisplayCurrency);
        return new GetByKeyRequestBuilder<BusPartInvoiceItemParameters>(
            servicePath,
            BusPartInvoiceItemParameters.class,
            key,
            "BusPartInvoiceItem");
    }

}
