/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingbusinesspartnerpayment.BusPartPaymentHeader;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingbusinesspartnerpayment.BusPartPaymentItemParameters;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingbusinesspartnerpayment.BusPartPaymentParameters;

import lombok.Getter;

/**
 * <p>
 * 
 * Retrieve a list of payments and make, release or withdraw payments. Read payment items and corresponding payment
 * headers for a business partner. Payment items are either document items which are cleared by payments, payments on
 * account or released payments. Post or release payments for an invoice or an open document item. A released payment
 * can also be withdrawn until the corresponding items are not cleared..
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/f8/8e28496bfa4bd281ea7054049ce341/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/API_CABUSPARTPAYMENT_0001?section=OVERVIEW'>SAP
 * API Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>API_CABUSPARTPAYMENT_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Finance - Contract Accounting Open Item Management Integration (SAP_COM_0361)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/2UJ'>Contract Accounting - Open Item Management (2UJ)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_CABUSPARTPAYMENT</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>ContrAcctgDocContainer</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultContractAccountingBusinessPartnerPaymentService" )
public class DefaultContractAccountingBusinessPartnerPaymentService
    implements
    ServiceWithNavigableEntities,
    ContractAccountingBusinessPartnerPaymentService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link ContractAccountingBusinessPartnerPaymentService#DEFAULT_SERVICE_PATH} to send the
     * requests.
     * 
     */
    public DefaultContractAccountingBusinessPartnerPaymentService()
    {
        servicePath = ContractAccountingBusinessPartnerPaymentService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultContractAccountingBusinessPartnerPaymentService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultContractAccountingBusinessPartnerPaymentService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultContractAccountingBusinessPartnerPaymentService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<BusPartPaymentParameters> getBusPartPaymentByKey(
        @Nonnull final String p_DisplayCurrency )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("P_DisplayCurrency", p_DisplayCurrency);
        return new GetByKeyRequestBuilder<BusPartPaymentParameters>(
            servicePath,
            BusPartPaymentParameters.class,
            key,
            "BusPartPayment");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<BusPartPaymentHeader> getAllBusPartPaymentHeader()
    {
        return new GetAllRequestBuilder<BusPartPaymentHeader>(
            servicePath,
            BusPartPaymentHeader.class,
            "BusPartPaymentHeader");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<BusPartPaymentHeader> countBusPartPaymentHeader()
    {
        return new CountRequestBuilder<BusPartPaymentHeader>(
            servicePath,
            BusPartPaymentHeader.class,
            "BusPartPaymentHeader");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<BusPartPaymentHeader> getBusPartPaymentHeaderByKey(
        @Nonnull final UUID cADocumentContainerPaymentUUID )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("CADocumentContainerPaymentUUID", cADocumentContainerPaymentUUID);
        return new GetByKeyRequestBuilder<BusPartPaymentHeader>(
            servicePath,
            BusPartPaymentHeader.class,
            key,
            "BusPartPaymentHeader");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<BusPartPaymentHeader> createBusPartPaymentHeader(
        @Nonnull final BusPartPaymentHeader busPartPaymentHeader )
    {
        return new CreateRequestBuilder<BusPartPaymentHeader>(
            servicePath,
            busPartPaymentHeader,
            "BusPartPaymentHeader");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<BusPartPaymentItemParameters> getBusPartPaymentItemByKey(
        @Nonnull final String p_DisplayCurrency )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("P_DisplayCurrency", p_DisplayCurrency);
        return new GetByKeyRequestBuilder<BusPartPaymentItemParameters>(
            servicePath,
            BusPartPaymentItemParameters.class,
            key,
            "BusPartPaymentItem");
    }

}
