/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingcreditworthiness.CreditWorthiness;

import lombok.Getter;

/**
 * <p>
 * 
 * This inbound OData service enables external systems and other consumers to retrieve creditworthiness data of business
 * partners.
 * 
 * This data includes, for example, the creditworthiness score, the creditworthiness date, and whether the
 * creditworthiness value is frozen for a business partner.
 * 
 * The service transfers a list of creditworthiness records for individual business partners.
 * 
 * This service is intended for the retrieval of information only..
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/09/18927e95464ab585175e73afb5a6b5/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/CACREDITWORTHINESS_0001?section=OVERVIEW'>SAP
 * API Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>CACREDITWORTHINESS_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Finance - Contract Accounting Integration (SAP_COM_0731)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/2UJ'>Contract Accounting - Open Item Management (2UJ)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_CACREDITWORTHINESS</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>ContrAcctgCreditWorthiness</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultContractAccountingCreditworthinessService" )
public class DefaultContractAccountingCreditworthinessService
    implements
    ServiceWithNavigableEntities,
    ContractAccountingCreditworthinessService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link ContractAccountingCreditworthinessService#DEFAULT_SERVICE_PATH} to send the
     * requests.
     * 
     */
    public DefaultContractAccountingCreditworthinessService()
    {
        servicePath = ContractAccountingCreditworthinessService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultContractAccountingCreditworthinessService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultContractAccountingCreditworthinessService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultContractAccountingCreditworthinessService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<CreditWorthiness> getAllCreditWorthiness()
    {
        return new GetAllRequestBuilder<CreditWorthiness>(servicePath, CreditWorthiness.class, "CreditWorthiness");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<CreditWorthiness> countCreditWorthiness()
    {
        return new CountRequestBuilder<CreditWorthiness>(servicePath, CreditWorthiness.class, "CreditWorthiness");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<CreditWorthiness> getCreditWorthinessByKey( @Nonnull final String businessPartner )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("BusinessPartner", businessPartner);
        return new GetByKeyRequestBuilder<CreditWorthiness>(
            servicePath,
            CreditWorthiness.class,
            key,
            "CreditWorthiness");
    }

}
