/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdisputecase.DisputeCase;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdisputecase.DisputeCaseObject;

import lombok.Getter;

/**
 * <p>
 * 
 * Create a dispute case with dispute case objects and assign attachments and notes to it. A dispute case object could
 * be a business partner, contract account, contract or FI-CA documents. Using the copy action, an existing dispute case
 * can be used as a template for the creation of a new dispute case.
 * 
 * # .
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/df/664ece25244b07ae7fff6d20d658c9/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/CADISPUTECASE_0001?section=OVERVIEW'>SAP API
 * Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>CADISPUTECASE_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Finance - Contract Accounting Integration (SAP_COM_0731)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/2UJ'>Contract Accounting - Open Item Management (2UJ)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_CADISPUTECASE</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>ContrAcctgDisputeCase</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultContractAccountingDisputeCaseService" )
public class DefaultContractAccountingDisputeCaseService
    implements
    ServiceWithNavigableEntities,
    ContractAccountingDisputeCaseService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link ContractAccountingDisputeCaseService#DEFAULT_SERVICE_PATH} to send the requests.
     * 
     */
    public DefaultContractAccountingDisputeCaseService()
    {
        servicePath = ContractAccountingDisputeCaseService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultContractAccountingDisputeCaseService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultContractAccountingDisputeCaseService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultContractAccountingDisputeCaseService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<DisputeCase> getAllDisputeCase()
    {
        return new GetAllRequestBuilder<DisputeCase>(servicePath, DisputeCase.class, "DisputeCase");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<DisputeCase> countDisputeCase()
    {
        return new CountRequestBuilder<DisputeCase>(servicePath, DisputeCase.class, "DisputeCase");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<DisputeCase> getDisputeCaseByKey( @Nonnull final UUID caseUUID )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("CaseUUID", caseUUID);
        return new GetByKeyRequestBuilder<DisputeCase>(servicePath, DisputeCase.class, key, "DisputeCase");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<DisputeCase> createDisputeCase( @Nonnull final DisputeCase disputeCase )
    {
        return new CreateRequestBuilder<DisputeCase>(servicePath, disputeCase, "DisputeCase");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<DisputeCaseObject> getAllDisputeCaseObject()
    {
        return new GetAllRequestBuilder<DisputeCaseObject>(servicePath, DisputeCaseObject.class, "DisputeCaseObject");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<DisputeCaseObject> countDisputeCaseObject()
    {
        return new CountRequestBuilder<DisputeCaseObject>(servicePath, DisputeCaseObject.class, "DisputeCaseObject");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<DisputeCaseObject> getDisputeCaseObjectByKey(
        @Nonnull final UUID caseUUID,
        @Nonnull final String objectType,
        @Nonnull final String objectKey )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("CaseUUID", caseUUID);
        key.put("ObjectType", objectType);
        key.put("ObjectKey", objectKey);
        return new GetByKeyRequestBuilder<DisputeCaseObject>(
            servicePath,
            DisputeCaseObject.class,
            key,
            "DisputeCaseObject");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<DisputeCaseObject> createDisputeCaseObject(
        @Nonnull final DisputeCaseObject disputeCaseObject )
    {
        return new CreateRequestBuilder<DisputeCaseObject>(servicePath, disputeCaseObject, "DisputeCaseObject");
    }

}
