/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.time.LocalDate;
import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.DeleteRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemBusLock;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemLogical;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.BPItemPaytData;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.DocumentHeader;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.GLItem;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.contractaccountingdocumentmanage.GLItemPaytData;

import lombok.Getter;

/**
 * <p>
 * 
 * Retrieve a list of documents. Read data on document header, document business partner line items and document general
 * ledger line items.
 * 
 * Update document header and document business partner line items. Create and delete business locks on existing
 * business partner line items..
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/7f/7caf54278145009e3c0e9678d8ffa6/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/CADOCUMENTMANAGE_0001?section=OVERVIEW'>SAP API
 * Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>CADOCUMENTMANAGE_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Finance - Contract Accounting Open Item Management Integration (SAP_COM_0361)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/2UJ'>Contract Accounting - Open Item Management (2UJ)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_CADOCUMENTMANAGE</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>ContrAcctgDocument</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultContractAccountingDocumentManageService" )
public class DefaultContractAccountingDocumentManageService
    implements
    ServiceWithNavigableEntities,
    ContractAccountingDocumentManageService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link ContractAccountingDocumentManageService#DEFAULT_SERVICE_PATH} to send the
     * requests.
     * 
     */
    public DefaultContractAccountingDocumentManageService()
    {
        servicePath = ContractAccountingDocumentManageService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultContractAccountingDocumentManageService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultContractAccountingDocumentManageService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultContractAccountingDocumentManageService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<BPItemBusLock> getAllBPItemBusLock()
    {
        return new GetAllRequestBuilder<BPItemBusLock>(servicePath, BPItemBusLock.class, "BPItemBusLock");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<BPItemBusLock> countBPItemBusLock()
    {
        return new CountRequestBuilder<BPItemBusLock>(servicePath, BPItemBusLock.class, "BPItemBusLock");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<BPItemBusLock> getBPItemBusLockByKey(
        @Nonnull final String cADocumentNumber,
        @Nonnull final String cARepetitionItemNumber,
        @Nonnull final String cABPItemNumber,
        @Nonnull final String cASubItemNumber,
        @Nonnull final String cABusinessProcess,
        @Nonnull final String cABusinessLockReason,
        @Nonnull final LocalDate cABusinessLockValidFrom,
        @Nonnull final LocalDate cABusinessLockValidTo )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("CADocumentNumber", cADocumentNumber);
        key.put("CARepetitionItemNumber", cARepetitionItemNumber);
        key.put("CABPItemNumber", cABPItemNumber);
        key.put("CASubItemNumber", cASubItemNumber);
        key.put("CABusinessProcess", cABusinessProcess);
        key.put("CABusinessLockReason", cABusinessLockReason);
        key.put("CABusinessLockValidFrom", cABusinessLockValidFrom);
        key.put("CABusinessLockValidTo", cABusinessLockValidTo);
        return new GetByKeyRequestBuilder<BPItemBusLock>(servicePath, BPItemBusLock.class, key, "BPItemBusLock");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<BPItemBusLock> createBPItemBusLock( @Nonnull final BPItemBusLock bPItemBusLock )
    {
        return new CreateRequestBuilder<BPItemBusLock>(servicePath, bPItemBusLock, "BPItemBusLock");
    }

    @Override
    @Nonnull
    public DeleteRequestBuilder<BPItemBusLock> deleteBPItemBusLock( @Nonnull final BPItemBusLock bPItemBusLock )
    {
        return new DeleteRequestBuilder<BPItemBusLock>(servicePath, bPItemBusLock, "BPItemBusLock");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<BPItemLogical> getAllBPItemLogical()
    {
        return new GetAllRequestBuilder<BPItemLogical>(servicePath, BPItemLogical.class, "BPItemLogical");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<BPItemLogical> countBPItemLogical()
    {
        return new CountRequestBuilder<BPItemLogical>(servicePath, BPItemLogical.class, "BPItemLogical");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<BPItemLogical> getBPItemLogicalByKey(
        @Nonnull final String cADocumentNumber,
        @Nonnull final String cARepetitionItemNumber,
        @Nonnull final String cABPItemNumber,
        @Nonnull final String cASubItemNumber )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("CADocumentNumber", cADocumentNumber);
        key.put("CARepetitionItemNumber", cARepetitionItemNumber);
        key.put("CABPItemNumber", cABPItemNumber);
        key.put("CASubItemNumber", cASubItemNumber);
        return new GetByKeyRequestBuilder<BPItemLogical>(servicePath, BPItemLogical.class, key, "BPItemLogical");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<BPItemLogical> createBPItemLogical( @Nonnull final BPItemLogical bPItemLogical )
    {
        return new CreateRequestBuilder<BPItemLogical>(servicePath, bPItemLogical, "BPItemLogical");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<BPItemLogical> updateBPItemLogical( @Nonnull final BPItemLogical bPItemLogical )
    {
        return new UpdateRequestBuilder<BPItemLogical>(servicePath, bPItemLogical, "BPItemLogical");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<BPItemPaytData> getAllBPItemPaytData()
    {
        return new GetAllRequestBuilder<BPItemPaytData>(servicePath, BPItemPaytData.class, "BPItemPaytData");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<BPItemPaytData> countBPItemPaytData()
    {
        return new CountRequestBuilder<BPItemPaytData>(servicePath, BPItemPaytData.class, "BPItemPaytData");
    }

    @Override
    @Nonnull
    public
        GetByKeyRequestBuilder<BPItemPaytData>
        getBPItemPaytDataByKey( @Nonnull final String cADocumentNumber, @Nonnull final String cAGroupingForPayment )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("CADocumentNumber", cADocumentNumber);
        key.put("CAGroupingForPayment", cAGroupingForPayment);
        return new GetByKeyRequestBuilder<BPItemPaytData>(servicePath, BPItemPaytData.class, key, "BPItemPaytData");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<DocumentHeader> getAllDocumentHeader()
    {
        return new GetAllRequestBuilder<DocumentHeader>(servicePath, DocumentHeader.class, "DocumentHeader");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<DocumentHeader> countDocumentHeader()
    {
        return new CountRequestBuilder<DocumentHeader>(servicePath, DocumentHeader.class, "DocumentHeader");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<DocumentHeader> getDocumentHeaderByKey( @Nonnull final String cADocumentNumber )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("CADocumentNumber", cADocumentNumber);
        return new GetByKeyRequestBuilder<DocumentHeader>(servicePath, DocumentHeader.class, key, "DocumentHeader");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<DocumentHeader> updateDocumentHeader( @Nonnull final DocumentHeader documentHeader )
    {
        return new UpdateRequestBuilder<DocumentHeader>(servicePath, documentHeader, "DocumentHeader");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<GLItem> getAllGLItem()
    {
        return new GetAllRequestBuilder<GLItem>(servicePath, GLItem.class, "GLItem");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<GLItem> countGLItem()
    {
        return new CountRequestBuilder<GLItem>(servicePath, GLItem.class, "GLItem");
    }

    @Override
    @Nonnull
    public
        GetByKeyRequestBuilder<GLItem>
        getGLItemByKey( @Nonnull final String cADocumentNumber, @Nonnull final String cAGLItemNumber )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("CADocumentNumber", cADocumentNumber);
        key.put("CAGLItemNumber", cAGLItemNumber);
        return new GetByKeyRequestBuilder<GLItem>(servicePath, GLItem.class, key, "GLItem");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<GLItemPaytData> getAllGLItemPaytData()
    {
        return new GetAllRequestBuilder<GLItemPaytData>(servicePath, GLItemPaytData.class, "GLItemPaytData");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<GLItemPaytData> countGLItemPaytData()
    {
        return new CountRequestBuilder<GLItemPaytData>(servicePath, GLItemPaytData.class, "GLItemPaytData");
    }

    @Override
    @Nonnull
    public
        GetByKeyRequestBuilder<GLItemPaytData>
        getGLItemPaytDataByKey( @Nonnull final String cADocumentNumber, @Nonnull final String cAGLItemNumber )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("CADocumentNumber", cADocumentNumber);
        key.put("CAGLItemNumber", cAGLItemNumber);
        return new GetByKeyRequestBuilder<GLItemPaytData>(servicePath, GLItemPaytData.class, key, "GLItemPaytData");
    }

}
