/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.documentcomplianceelectronicdocumentfile.ElectronicDocFile;

import lombok.Getter;

/**
 * <p>
 * This service enables you to download eDocument file data in SAP S/4HANA Cloud that can be used by external
 * applications. The service is based on the OData V4 protocol.
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/2a/0e6664329a457590422ca9f042bed3/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/ELECTRONICDOCFILE_0001?section=OVERVIEW'>SAP
 * API Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>ELECTRONICDOCFILE_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Document Compliance - Electronic Document File Integration (SAP_COM_0703)</td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_ELECTRONICDOCFILE</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>ElectronicDocument</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultDocumentComplianceElectronicDocumentFileService" )
public class DefaultDocumentComplianceElectronicDocumentFileService
    implements
    ServiceWithNavigableEntities,
    DocumentComplianceElectronicDocumentFileService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link DocumentComplianceElectronicDocumentFileService#DEFAULT_SERVICE_PATH} to send the
     * requests.
     * 
     */
    public DefaultDocumentComplianceElectronicDocumentFileService()
    {
        servicePath = DocumentComplianceElectronicDocumentFileService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultDocumentComplianceElectronicDocumentFileService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultDocumentComplianceElectronicDocumentFileService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultDocumentComplianceElectronicDocumentFileService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<ElectronicDocFile> getAllElectronicDocFile()
    {
        return new GetAllRequestBuilder<ElectronicDocFile>(servicePath, ElectronicDocFile.class, "ElectronicDocFile");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<ElectronicDocFile> countElectronicDocFile()
    {
        return new CountRequestBuilder<ElectronicDocFile>(servicePath, ElectronicDocFile.class, "ElectronicDocFile");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<ElectronicDocFile> getElectronicDocFileByKey(
        @Nonnull final UUID electronicDocFileUUID )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("ElectronicDocFileUUID", electronicDocFileUUID);
        return new GetByKeyRequestBuilder<ElectronicDocFile>(
            servicePath,
            ElectronicDocFile.class,
            key,
            "ElectronicDocFile");
    }

}
