/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.enterpriseprojectinternaltype.EntProjectType;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.enterpriseprojectinternaltype.EntProjectTypeText;

import lombok.Getter;

/**
 * <p>
 * This service enables you to read the internal project type code list for enterprise projects. You can use it to make
 * the code list values available for the corresponding Enterprise Project service.
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/3c/cfae81e2674b9f8464bd1d637fa869/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/ENTPROJECTTYPE_0001?section=OVERVIEW'>SAP API
 * Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>ENTPROJECTTYPE_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Enterprise Project - Internal Project Type Integration (SAP_COM_0723)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/1NT'>Project Financial Control (1NT)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/1YF'>Project Review (1YF)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509, OAuth2</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_ENTERPRISEPROJECTTYPE</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>EnterpriseProject</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultEnterpriseProjectInternalTypeService" )
public class DefaultEnterpriseProjectInternalTypeService
    implements
    ServiceWithNavigableEntities,
    EnterpriseProjectInternalTypeService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link EnterpriseProjectInternalTypeService#DEFAULT_SERVICE_PATH} to send the requests.
     * 
     */
    public DefaultEnterpriseProjectInternalTypeService()
    {
        servicePath = EnterpriseProjectInternalTypeService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultEnterpriseProjectInternalTypeService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultEnterpriseProjectInternalTypeService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultEnterpriseProjectInternalTypeService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<EntProjectType> getAllEntProjectType()
    {
        return new GetAllRequestBuilder<EntProjectType>(servicePath, EntProjectType.class, "EntProjectType");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<EntProjectType> countEntProjectType()
    {
        return new CountRequestBuilder<EntProjectType>(servicePath, EntProjectType.class, "EntProjectType");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<EntProjectType> getEntProjectTypeByKey( @Nonnull final String entProjectTypeInternal )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("EntProjectTypeInternal", entProjectTypeInternal);
        return new GetByKeyRequestBuilder<EntProjectType>(servicePath, EntProjectType.class, key, "EntProjectType");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<EntProjectTypeText> getAllEntProjectTypeText()
    {
        return new GetAllRequestBuilder<EntProjectTypeText>(
            servicePath,
            EntProjectTypeText.class,
            "EntProjectTypeText");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<EntProjectTypeText> countEntProjectTypeText()
    {
        return new CountRequestBuilder<EntProjectTypeText>(servicePath, EntProjectTypeText.class, "EntProjectTypeText");
    }

    @Override
    @Nonnull
    public
        GetByKeyRequestBuilder<EntProjectTypeText>
        getEntProjectTypeTextByKey( @Nonnull final String language, @Nonnull final String entProjectTypeInternal )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("Language", language);
        key.put("EntProjectTypeInternal", entProjectTypeInternal);
        return new GetByKeyRequestBuilder<EntProjectTypeText>(
            servicePath,
            EntProjectTypeText.class,
            key,
            "EntProjectTypeText");
    }

}
