/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.enterpriseprojectpriority.EntProjPriorityCode;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.enterpriseprojectpriority.EntProjPriorityCodeText;

import lombok.Getter;

/**
 * <p>
 * This service enables you to read the processing status code list for enterprise projects. You can use it to make the
 * code list values available for the corresponding Enterprise Project service.
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/08/4d5535449b49d7b52da079b60b9587/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/ENTPROJECTPRIORITYCODE_0001?section=OVERVIEW'>SAP
 * API Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>ENTPROJECTPRIORITYCODE_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Enterprise Project - Project Priority Integration (SAP_COM_0726)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/1NT'>Project Financial Control (1NT)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/1YF'>Project Review (1YF)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509, OAuth2</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_ENTPROJPRIORITYCODE</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>EnterpriseProject</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultEnterpriseProjectPriorityService" )
public class DefaultEnterpriseProjectPriorityService
    implements
    ServiceWithNavigableEntities,
    EnterpriseProjectPriorityService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link EnterpriseProjectPriorityService#DEFAULT_SERVICE_PATH} to send the requests.
     * 
     */
    public DefaultEnterpriseProjectPriorityService()
    {
        servicePath = EnterpriseProjectPriorityService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultEnterpriseProjectPriorityService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultEnterpriseProjectPriorityService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultEnterpriseProjectPriorityService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<EntProjPriorityCode> getAllEntProjPriorityCode()
    {
        return new GetAllRequestBuilder<EntProjPriorityCode>(
            servicePath,
            EntProjPriorityCode.class,
            "EntProjPriorityCode");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<EntProjPriorityCode> countEntProjPriorityCode()
    {
        return new CountRequestBuilder<EntProjPriorityCode>(
            servicePath,
            EntProjPriorityCode.class,
            "EntProjPriorityCode");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<EntProjPriorityCode> getEntProjPriorityCodeByKey( @Nonnull final String priorityCode )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("PriorityCode", priorityCode);
        return new GetByKeyRequestBuilder<EntProjPriorityCode>(
            servicePath,
            EntProjPriorityCode.class,
            key,
            "EntProjPriorityCode");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<EntProjPriorityCodeText> getAllEntProjPriorityCodeText()
    {
        return new GetAllRequestBuilder<EntProjPriorityCodeText>(
            servicePath,
            EntProjPriorityCodeText.class,
            "EntProjPriorityCodeText");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<EntProjPriorityCodeText> countEntProjPriorityCodeText()
    {
        return new CountRequestBuilder<EntProjPriorityCodeText>(
            servicePath,
            EntProjPriorityCodeText.class,
            "EntProjPriorityCodeText");
    }

    @Override
    @Nonnull
    public
        GetByKeyRequestBuilder<EntProjPriorityCodeText>
        getEntProjPriorityCodeTextByKey( @Nonnull final String language, @Nonnull final String priorityCode )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("Language", language);
        key.put("PriorityCode", priorityCode);
        return new GetByKeyRequestBuilder<EntProjPriorityCodeText>(
            servicePath,
            EntProjPriorityCodeText.class,
            key,
            "EntProjPriorityCodeText");
    }

}
