/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.enterpriseprojectprocessingstatus.ProcessingStatus;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.enterpriseprojectprocessingstatus.ProcessingStatusText;

import lombok.Getter;

/**
 * <p>
 * This service enables you to read the processing status code list for enterprise projects. You can use it to make the
 * code list values available for the corresponding Enterprise Project service.
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/09/0091bf7f0e46bcbce23f2ebda8905d/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/ENTPROJECTPROCESSINGSTATUS_0001?section=OVERVIEW'>SAP
 * API Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>ENTPROJECTPROCESSINGSTATUS_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Enterprise Project - Project Processing Status Integration (SAP_COM_0725)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/1NT'>Project Financial Control (1NT)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/1YF'>Project Review (1YF)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509, OAuth2</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_ENTPROJPROCESSINGSTAT</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>EnterpriseProject</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultEnterpriseProjectProcessingStatusService" )
public class DefaultEnterpriseProjectProcessingStatusService
    implements
    ServiceWithNavigableEntities,
    EnterpriseProjectProcessingStatusService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link EnterpriseProjectProcessingStatusService#DEFAULT_SERVICE_PATH} to send the
     * requests.
     * 
     */
    public DefaultEnterpriseProjectProcessingStatusService()
    {
        servicePath = EnterpriseProjectProcessingStatusService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultEnterpriseProjectProcessingStatusService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultEnterpriseProjectProcessingStatusService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultEnterpriseProjectProcessingStatusService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<ProcessingStatus> getAllProcessingStatus()
    {
        return new GetAllRequestBuilder<ProcessingStatus>(servicePath, ProcessingStatus.class, "ProcessingStatus");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<ProcessingStatus> countProcessingStatus()
    {
        return new CountRequestBuilder<ProcessingStatus>(servicePath, ProcessingStatus.class, "ProcessingStatus");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<ProcessingStatus> getProcessingStatusByKey( @Nonnull final String processingStatus )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("ProcessingStatus", processingStatus);
        return new GetByKeyRequestBuilder<ProcessingStatus>(
            servicePath,
            ProcessingStatus.class,
            key,
            "ProcessingStatus");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<ProcessingStatusText> getAllProcessingStatusText()
    {
        return new GetAllRequestBuilder<ProcessingStatusText>(
            servicePath,
            ProcessingStatusText.class,
            "ProcessingStatusText");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<ProcessingStatusText> countProcessingStatusText()
    {
        return new CountRequestBuilder<ProcessingStatusText>(
            servicePath,
            ProcessingStatusText.class,
            "ProcessingStatusText");
    }

    @Override
    @Nonnull
    public
        GetByKeyRequestBuilder<ProcessingStatusText>
        getProcessingStatusTextByKey( @Nonnull final String language, @Nonnull final String processingStatus )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("Language", language);
        key.put("ProcessingStatus", processingStatus);
        return new GetByKeyRequestBuilder<ProcessingStatusText>(
            servicePath,
            ProcessingStatusText.class,
            key,
            "ProcessingStatusText");
    }

}
