/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.enterpriseprojectprofile.ProjectProfileCode;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.enterpriseprojectprofile.ProjectProfileCodeText;

import lombok.Getter;

/**
 * <p>
 * This service enables you to read the project profile code list for enterprise projects. You can use it to make the
 * code list values available for the corresponding Enterprise Project service.
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/9f/8ed7a355eb4d0cb8c5282b138b7889/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/ENTPROJECTPROFILECODE_0001?section=OVERVIEW'>SAP
 * API Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>ENTPROJECTPROFILECODE_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Enterprise Project - Project Profile Integration (SAP_COM_0724)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/1NT'>Project Financial Control (1NT)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/1YF'>Project Review (1YF)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509, OAuth2</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_ENTPROJECTPROFILECODE</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>EnterpriseProject</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultEnterpriseProjectProfileService" )
public class DefaultEnterpriseProjectProfileService
    implements
    ServiceWithNavigableEntities,
    EnterpriseProjectProfileService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link EnterpriseProjectProfileService#DEFAULT_SERVICE_PATH} to send the requests.
     * 
     */
    public DefaultEnterpriseProjectProfileService()
    {
        servicePath = EnterpriseProjectProfileService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultEnterpriseProjectProfileService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultEnterpriseProjectProfileService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultEnterpriseProjectProfileService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<ProjectProfileCode> getAllProjectProfileCode()
    {
        return new GetAllRequestBuilder<ProjectProfileCode>(
            servicePath,
            ProjectProfileCode.class,
            "ProjectProfileCode");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<ProjectProfileCode> countProjectProfileCode()
    {
        return new CountRequestBuilder<ProjectProfileCode>(servicePath, ProjectProfileCode.class, "ProjectProfileCode");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<ProjectProfileCode> getProjectProfileCodeByKey(
        @Nonnull final String projectProfileCode )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("ProjectProfileCode", projectProfileCode);
        return new GetByKeyRequestBuilder<ProjectProfileCode>(
            servicePath,
            ProjectProfileCode.class,
            key,
            "ProjectProfileCode");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<ProjectProfileCodeText> getAllProjectProfileCodeText()
    {
        return new GetAllRequestBuilder<ProjectProfileCodeText>(
            servicePath,
            ProjectProfileCodeText.class,
            "ProjectProfileCodeText");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<ProjectProfileCodeText> countProjectProfileCodeText()
    {
        return new CountRequestBuilder<ProjectProfileCodeText>(
            servicePath,
            ProjectProfileCodeText.class,
            "ProjectProfileCodeText");
    }

    @Override
    @Nonnull
    public
        GetByKeyRequestBuilder<ProjectProfileCodeText>
        getProjectProfileCodeTextByKey( @Nonnull final String language, @Nonnull final String projectProfileCode )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("Language", language);
        key.put("ProjectProfileCode", projectProfileCode);
        return new GetByKeyRequestBuilder<ProjectProfileCodeText>(
            servicePath,
            ProjectProfileCodeText.class,
            key,
            "ProjectProfileCodeText");
    }

}
