/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.environmenthealthandsafetyexternalsource.EHSAmountExternalSource;

import lombok.Getter;

/**
 * <p>
 * 
 * This service enables you to read available External Source IDs and send amounts to the system. You can send multiple
 * amounts via a changeset..
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/a6/54bb3526e34b7abf4450d08eb4ca50/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/EHSAMOUNTEXTERNALSRCE_0001?section=OVERVIEW'>SAP
 * API Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>EHSAMOUNTEXTERNALSRCE_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Environment, Health, and Safety - Automatic Data Collection Integration (SAP_COM_0747)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/4XD'>Emissions Management (4XD)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_EHSAMOUNTEXTERNALSRCE</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>EHSDataCollection</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultEnvironmentHealthAndSafetyExternalSourceService" )
public class DefaultEnvironmentHealthAndSafetyExternalSourceService
    implements
    ServiceWithNavigableEntities,
    EnvironmentHealthAndSafetyExternalSourceService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link EnvironmentHealthAndSafetyExternalSourceService#DEFAULT_SERVICE_PATH} to send the
     * requests.
     * 
     */
    public DefaultEnvironmentHealthAndSafetyExternalSourceService()
    {
        servicePath = EnvironmentHealthAndSafetyExternalSourceService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultEnvironmentHealthAndSafetyExternalSourceService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultEnvironmentHealthAndSafetyExternalSourceService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultEnvironmentHealthAndSafetyExternalSourceService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<EHSAmountExternalSource> getAllEHSAmountExternalSource()
    {
        return new GetAllRequestBuilder<EHSAmountExternalSource>(
            servicePath,
            EHSAmountExternalSource.class,
            "EHSAmountExternalSource");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<EHSAmountExternalSource> countEHSAmountExternalSource()
    {
        return new CountRequestBuilder<EHSAmountExternalSource>(
            servicePath,
            EHSAmountExternalSource.class,
            "EHSAmountExternalSource");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<EHSAmountExternalSource> getEHSAmountExternalSourceByKey(
        @Nonnull final String eHSAmountExternalSource )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("EHSAmountExternalSource", eHSAmountExternalSource);
        return new GetByKeyRequestBuilder<EHSAmountExternalSource>(
            servicePath,
            EHSAmountExternalSource.class,
            key,
            "EHSAmountExternalSource");
    }

}
