/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.handlingunit.HandlingUnit;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.handlingunit.HandlingUnitItem;

import lombok.Getter;

/**
 * <p>
 * 
 * The service contains handling unit header and item nodes. With this API, you can carry out the following operations:
 * - Retrieve header and item information from handling units, for example, displaying all handling for a plant, or
 * displaying the products packed in a specific handling unit. For handling units processed in Warehouse Management,
 * warehouse number and handling unit number must be used as a key. For handling units not relevant for Warehouse
 * Management, the handling unit number is used as a key. - Create handling units without reference to a document. These
 * non-assigned handling units are used, for example, when the physical creation of a handling unit is planned or when a
 * handling unit already exists physically but is not yet entered in the system. - Post goods movement for handling
 * units. You can move handling units to different plants and to different storage locations within one plant and
 * transfer handling units from an SAP S/4HANA storage location to a Warehouse Management storage bin. .
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/08/31a7112057480a8f61b8bf01b56202/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference:
 * <a href='https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/HANDLINGUNIT_0001?section=OVERVIEW'>SAP
 * API Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>HANDLINGUNIT_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Handling Unit Integration (SAP_COM_0364)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/3BS'>Warehouse Outbound Processing (3BS)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BD9'>Sell from Stock (BD9)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_HANDLINGUNIT</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>HandlingUnit</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultHandlingUnitService" )
public class DefaultHandlingUnitService implements ServiceWithNavigableEntities, HandlingUnitService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link HandlingUnitService#DEFAULT_SERVICE_PATH} to send the requests.
     * 
     */
    public DefaultHandlingUnitService()
    {
        servicePath = HandlingUnitService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultHandlingUnitService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultHandlingUnitService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultHandlingUnitService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<HandlingUnit> getAllHandlingUnit()
    {
        return new GetAllRequestBuilder<HandlingUnit>(servicePath, HandlingUnit.class, "HandlingUnit");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<HandlingUnit> countHandlingUnit()
    {
        return new CountRequestBuilder<HandlingUnit>(servicePath, HandlingUnit.class, "HandlingUnit");
    }

    @Override
    @Nonnull
    public
        GetByKeyRequestBuilder<HandlingUnit>
        getHandlingUnitByKey( @Nonnull final String handlingUnitExternalID, @Nonnull final String warehouse )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("HandlingUnitExternalID", handlingUnitExternalID);
        key.put("Warehouse", warehouse);
        return new GetByKeyRequestBuilder<HandlingUnit>(servicePath, HandlingUnit.class, key, "HandlingUnit");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<HandlingUnit> createHandlingUnit( @Nonnull final HandlingUnit handlingUnit )
    {
        return new CreateRequestBuilder<HandlingUnit>(servicePath, handlingUnit, "HandlingUnit");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<HandlingUnitItem> getAllHandlingUnitItem()
    {
        return new GetAllRequestBuilder<HandlingUnitItem>(servicePath, HandlingUnitItem.class, "HandlingUnitItem");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<HandlingUnitItem> countHandlingUnitItem()
    {
        return new CountRequestBuilder<HandlingUnitItem>(servicePath, HandlingUnitItem.class, "HandlingUnitItem");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<HandlingUnitItem> getHandlingUnitItemByKey(
        @Nonnull final String handlingUnitExternalID,
        @Nonnull final String warehouse,
        @Nonnull final UUID stockItemUUID )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("HandlingUnitExternalID", handlingUnitExternalID);
        key.put("Warehouse", warehouse);
        key.put("StockItemUUID", stockItemUUID);
        return new GetByKeyRequestBuilder<HandlingUnitItem>(
            servicePath,
            HandlingUnitItem.class,
            key,
            "HandlingUnitItem");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<HandlingUnitItem> createHandlingUnitItem(
        @Nonnull final HandlingUnitItem handlingUnitItem )
    {
        return new CreateRequestBuilder<HandlingUnitItem>(servicePath, handlingUnitItem, "HandlingUnitItem");
    }

}
