/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.DeleteRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.inhouserepair.InHouseRepair;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.inhouserepair.InHouseRepairItem;

import lombok.Getter;

/**
 * <p>
 * 
 * The service enables you to initiate the in-house repair process from an external system. You can create in-house
 * repairs and add repair objects. You can also read and update in-house repairs..
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/40/347a9ad1b54a4b931ea07b54faa42d/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/INHOUSEREPAIR_0001?section=OVERVIEW'>SAP API
 * Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>INHOUSEREPAIR_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Service - In-House Repair OData Integration (SAP_COM_0717)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/3XK'>In-House Repair (3XK)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509, OAuth2</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_INHOUSEREPAIR</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>InHouseRepair</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultInHouseRepairService" )
public class DefaultInHouseRepairService implements ServiceWithNavigableEntities, InHouseRepairService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link InHouseRepairService#DEFAULT_SERVICE_PATH} to send the requests.
     * 
     */
    public DefaultInHouseRepairService()
    {
        servicePath = InHouseRepairService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultInHouseRepairService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultInHouseRepairService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultInHouseRepairService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<InHouseRepair> getAllInHouseRepair()
    {
        return new GetAllRequestBuilder<InHouseRepair>(servicePath, InHouseRepair.class, "InHouseRepair");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<InHouseRepair> countInHouseRepair()
    {
        return new CountRequestBuilder<InHouseRepair>(servicePath, InHouseRepair.class, "InHouseRepair");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<InHouseRepair> getInHouseRepairByKey( @Nonnull final String inHouseRepair )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("InHouseRepair", inHouseRepair);
        return new GetByKeyRequestBuilder<InHouseRepair>(servicePath, InHouseRepair.class, key, "InHouseRepair");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<InHouseRepair> createInHouseRepair( @Nonnull final InHouseRepair inHouseRepair )
    {
        return new CreateRequestBuilder<InHouseRepair>(servicePath, inHouseRepair, "InHouseRepair");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<InHouseRepair> updateInHouseRepair( @Nonnull final InHouseRepair inHouseRepair )
    {
        return new UpdateRequestBuilder<InHouseRepair>(servicePath, inHouseRepair, "InHouseRepair");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<InHouseRepairItem> getAllInHouseRepairItem()
    {
        return new GetAllRequestBuilder<InHouseRepairItem>(servicePath, InHouseRepairItem.class, "InHouseRepairItem");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<InHouseRepairItem> countInHouseRepairItem()
    {
        return new CountRequestBuilder<InHouseRepairItem>(servicePath, InHouseRepairItem.class, "InHouseRepairItem");
    }

    @Override
    @Nonnull
    public
        GetByKeyRequestBuilder<InHouseRepairItem>
        getInHouseRepairItemByKey( @Nonnull final String inHouseRepair, @Nonnull final String inHouseRepairItem )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("InHouseRepair", inHouseRepair);
        key.put("InHouseRepairItem", inHouseRepairItem);
        return new GetByKeyRequestBuilder<InHouseRepairItem>(
            servicePath,
            InHouseRepairItem.class,
            key,
            "InHouseRepairItem");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<InHouseRepairItem> createInHouseRepairItem(
        @Nonnull final InHouseRepairItem inHouseRepairItem )
    {
        return new CreateRequestBuilder<InHouseRepairItem>(servicePath, inHouseRepairItem, "InHouseRepairItem");
    }

    @Override
    @Nonnull
    public DeleteRequestBuilder<InHouseRepairItem> deleteInHouseRepairItem(
        @Nonnull final InHouseRepairItem inHouseRepairItem )
    {
        return new DeleteRequestBuilder<InHouseRepairItem>(servicePath, inHouseRepairItem, "InHouseRepairItem");
    }

}
