/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.DeleteRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointoperatingagreement.JntOpgAgrmt;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointoperatingagreement.JntOpgAgrmtEquityTypeGrp;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointoperatingagreement.JntOpgAgrmtPartShr;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointoperatingagreement.JntOpgAgrmtPnltyCatRcvry;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointoperatingagreement.JntOpgAgrmtText;

import lombok.Getter;

/**
 * <p>
 * The service contains Joint Operating Agreement, Description, Equity Group, Penalty Category Recovery Percentage and
 * Partner Share nodes. The information is sent in the request as a payload, which is used in creating or updating a
 * Joint Operating Agreement, including appropriate checks for restricted values and authorizations. If there are any
 * issues when the Joint Operating Agreement is created, updated or deleted, the system displays error messages in the
 * response.
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/84/4ec46a9ed5480cbde49aec559c37ff/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/API_JNTOPGAGRMT_0001?section=OVERVIEW'>SAP API
 * Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>API_JNTOPGAGRMT_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Joint Venture Accounting API Integration (SAP_COM_0383)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/3F7'>Joint Venture Accounting (3F7)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_JNTOPGAGRMT</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultJointOperatingAgreementService" )
public class DefaultJointOperatingAgreementService
    implements
    ServiceWithNavigableEntities,
    JointOperatingAgreementService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link JointOperatingAgreementService#DEFAULT_SERVICE_PATH} to send the requests.
     * 
     */
    public DefaultJointOperatingAgreementService()
    {
        servicePath = JointOperatingAgreementService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultJointOperatingAgreementService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultJointOperatingAgreementService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultJointOperatingAgreementService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<JntOpgAgrmt> getAllJntOpgAgrmt()
    {
        return new GetAllRequestBuilder<JntOpgAgrmt>(servicePath, JntOpgAgrmt.class, "A_JntOpgAgrmt");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<JntOpgAgrmt> countJntOpgAgrmt()
    {
        return new CountRequestBuilder<JntOpgAgrmt>(servicePath, JntOpgAgrmt.class, "A_JntOpgAgrmt");
    }

    @Override
    @Nonnull
    public
        GetByKeyRequestBuilder<JntOpgAgrmt>
        getJntOpgAgrmtByKey( @Nonnull final String companyCode, @Nonnull final String jntOpgAgrmt )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("CompanyCode", companyCode);
        key.put("JntOpgAgrmt", jntOpgAgrmt);
        return new GetByKeyRequestBuilder<JntOpgAgrmt>(servicePath, JntOpgAgrmt.class, key, "A_JntOpgAgrmt");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<JntOpgAgrmt> createJntOpgAgrmt( @Nonnull final JntOpgAgrmt jntOpgAgrmt )
    {
        return new CreateRequestBuilder<JntOpgAgrmt>(servicePath, jntOpgAgrmt, "A_JntOpgAgrmt");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<JntOpgAgrmt> updateJntOpgAgrmt( @Nonnull final JntOpgAgrmt jntOpgAgrmt )
    {
        return new UpdateRequestBuilder<JntOpgAgrmt>(servicePath, jntOpgAgrmt, "A_JntOpgAgrmt");
    }

    @Override
    @Nonnull
    public DeleteRequestBuilder<JntOpgAgrmt> deleteJntOpgAgrmt( @Nonnull final JntOpgAgrmt jntOpgAgrmt )
    {
        return new DeleteRequestBuilder<JntOpgAgrmt>(servicePath, jntOpgAgrmt, "A_JntOpgAgrmt");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<JntOpgAgrmtEquityTypeGrp> getAllJntOpgAgrmtEquityTypeGrp()
    {
        return new GetAllRequestBuilder<JntOpgAgrmtEquityTypeGrp>(
            servicePath,
            JntOpgAgrmtEquityTypeGrp.class,
            "A_JntOpgAgrmtEquityTypeGrp");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<JntOpgAgrmtEquityTypeGrp> countJntOpgAgrmtEquityTypeGrp()
    {
        return new CountRequestBuilder<JntOpgAgrmtEquityTypeGrp>(
            servicePath,
            JntOpgAgrmtEquityTypeGrp.class,
            "A_JntOpgAgrmtEquityTypeGrp");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<JntOpgAgrmtEquityTypeGrp> getJntOpgAgrmtEquityTypeGrpByKey(
        @Nonnull final String companyCode,
        @Nonnull final String jntOpgAgrmt,
        @Nonnull final String jntOpgAgrmtEquityGrp )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("CompanyCode", companyCode);
        key.put("JntOpgAgrmt", jntOpgAgrmt);
        key.put("JntOpgAgrmtEquityGrp", jntOpgAgrmtEquityGrp);
        return new GetByKeyRequestBuilder<JntOpgAgrmtEquityTypeGrp>(
            servicePath,
            JntOpgAgrmtEquityTypeGrp.class,
            key,
            "A_JntOpgAgrmtEquityTypeGrp");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<JntOpgAgrmtEquityTypeGrp> createJntOpgAgrmtEquityTypeGrp(
        @Nonnull final JntOpgAgrmtEquityTypeGrp jntOpgAgrmtEquityTypeGrp )
    {
        return new CreateRequestBuilder<JntOpgAgrmtEquityTypeGrp>(
            servicePath,
            jntOpgAgrmtEquityTypeGrp,
            "A_JntOpgAgrmtEquityTypeGrp");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<JntOpgAgrmtEquityTypeGrp> updateJntOpgAgrmtEquityTypeGrp(
        @Nonnull final JntOpgAgrmtEquityTypeGrp jntOpgAgrmtEquityTypeGrp )
    {
        return new UpdateRequestBuilder<JntOpgAgrmtEquityTypeGrp>(
            servicePath,
            jntOpgAgrmtEquityTypeGrp,
            "A_JntOpgAgrmtEquityTypeGrp");
    }

    @Override
    @Nonnull
    public DeleteRequestBuilder<JntOpgAgrmtEquityTypeGrp> deleteJntOpgAgrmtEquityTypeGrp(
        @Nonnull final JntOpgAgrmtEquityTypeGrp jntOpgAgrmtEquityTypeGrp )
    {
        return new DeleteRequestBuilder<JntOpgAgrmtEquityTypeGrp>(
            servicePath,
            jntOpgAgrmtEquityTypeGrp,
            "A_JntOpgAgrmtEquityTypeGrp");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<JntOpgAgrmtPartShr> getAllJntOpgAgrmtPartShr()
    {
        return new GetAllRequestBuilder<JntOpgAgrmtPartShr>(
            servicePath,
            JntOpgAgrmtPartShr.class,
            "A_JntOpgAgrmtPartShr");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<JntOpgAgrmtPartShr> countJntOpgAgrmtPartShr()
    {
        return new CountRequestBuilder<JntOpgAgrmtPartShr>(
            servicePath,
            JntOpgAgrmtPartShr.class,
            "A_JntOpgAgrmtPartShr");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<JntOpgAgrmtPartShr> getJntOpgAgrmtPartShrByKey(
        @Nonnull final String companyCode,
        @Nonnull final String jntOpgAgrmt,
        @Nonnull final String jntOpgAgrmtEquityGrp,
        @Nonnull final String jointVenturePartner )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("CompanyCode", companyCode);
        key.put("JntOpgAgrmt", jntOpgAgrmt);
        key.put("JntOpgAgrmtEquityGrp", jntOpgAgrmtEquityGrp);
        key.put("JointVenturePartner", jointVenturePartner);
        return new GetByKeyRequestBuilder<JntOpgAgrmtPartShr>(
            servicePath,
            JntOpgAgrmtPartShr.class,
            key,
            "A_JntOpgAgrmtPartShr");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<JntOpgAgrmtPartShr> createJntOpgAgrmtPartShr(
        @Nonnull final JntOpgAgrmtPartShr jntOpgAgrmtPartShr )
    {
        return new CreateRequestBuilder<JntOpgAgrmtPartShr>(servicePath, jntOpgAgrmtPartShr, "A_JntOpgAgrmtPartShr");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<JntOpgAgrmtPartShr> updateJntOpgAgrmtPartShr(
        @Nonnull final JntOpgAgrmtPartShr jntOpgAgrmtPartShr )
    {
        return new UpdateRequestBuilder<JntOpgAgrmtPartShr>(servicePath, jntOpgAgrmtPartShr, "A_JntOpgAgrmtPartShr");
    }

    @Override
    @Nonnull
    public DeleteRequestBuilder<JntOpgAgrmtPartShr> deleteJntOpgAgrmtPartShr(
        @Nonnull final JntOpgAgrmtPartShr jntOpgAgrmtPartShr )
    {
        return new DeleteRequestBuilder<JntOpgAgrmtPartShr>(servicePath, jntOpgAgrmtPartShr, "A_JntOpgAgrmtPartShr");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<JntOpgAgrmtPnltyCatRcvry> getAllJntOpgAgrmtPnltyCatRcvry()
    {
        return new GetAllRequestBuilder<JntOpgAgrmtPnltyCatRcvry>(
            servicePath,
            JntOpgAgrmtPnltyCatRcvry.class,
            "A_JntOpgAgrmtPnltyCatRcvry");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<JntOpgAgrmtPnltyCatRcvry> countJntOpgAgrmtPnltyCatRcvry()
    {
        return new CountRequestBuilder<JntOpgAgrmtPnltyCatRcvry>(
            servicePath,
            JntOpgAgrmtPnltyCatRcvry.class,
            "A_JntOpgAgrmtPnltyCatRcvry");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<JntOpgAgrmtPnltyCatRcvry> getJntOpgAgrmtPnltyCatRcvryByKey(
        @Nonnull final String companyCode,
        @Nonnull final String jntOpgAgrmt,
        @Nonnull final String jntVntrPenaltyCategory )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("CompanyCode", companyCode);
        key.put("JntOpgAgrmt", jntOpgAgrmt);
        key.put("JntVntrPenaltyCategory", jntVntrPenaltyCategory);
        return new GetByKeyRequestBuilder<JntOpgAgrmtPnltyCatRcvry>(
            servicePath,
            JntOpgAgrmtPnltyCatRcvry.class,
            key,
            "A_JntOpgAgrmtPnltyCatRcvry");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<JntOpgAgrmtPnltyCatRcvry> createJntOpgAgrmtPnltyCatRcvry(
        @Nonnull final JntOpgAgrmtPnltyCatRcvry jntOpgAgrmtPnltyCatRcvry )
    {
        return new CreateRequestBuilder<JntOpgAgrmtPnltyCatRcvry>(
            servicePath,
            jntOpgAgrmtPnltyCatRcvry,
            "A_JntOpgAgrmtPnltyCatRcvry");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<JntOpgAgrmtPnltyCatRcvry> updateJntOpgAgrmtPnltyCatRcvry(
        @Nonnull final JntOpgAgrmtPnltyCatRcvry jntOpgAgrmtPnltyCatRcvry )
    {
        return new UpdateRequestBuilder<JntOpgAgrmtPnltyCatRcvry>(
            servicePath,
            jntOpgAgrmtPnltyCatRcvry,
            "A_JntOpgAgrmtPnltyCatRcvry");
    }

    @Override
    @Nonnull
    public DeleteRequestBuilder<JntOpgAgrmtPnltyCatRcvry> deleteJntOpgAgrmtPnltyCatRcvry(
        @Nonnull final JntOpgAgrmtPnltyCatRcvry jntOpgAgrmtPnltyCatRcvry )
    {
        return new DeleteRequestBuilder<JntOpgAgrmtPnltyCatRcvry>(
            servicePath,
            jntOpgAgrmtPnltyCatRcvry,
            "A_JntOpgAgrmtPnltyCatRcvry");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<JntOpgAgrmtText> getAllJntOpgAgrmtText()
    {
        return new GetAllRequestBuilder<JntOpgAgrmtText>(servicePath, JntOpgAgrmtText.class, "A_JntOpgAgrmtText");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<JntOpgAgrmtText> countJntOpgAgrmtText()
    {
        return new CountRequestBuilder<JntOpgAgrmtText>(servicePath, JntOpgAgrmtText.class, "A_JntOpgAgrmtText");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<JntOpgAgrmtText> getJntOpgAgrmtTextByKey(
        @Nonnull final String companyCode,
        @Nonnull final String language,
        @Nonnull final String jntOpgAgrmt )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("CompanyCode", companyCode);
        key.put("Language", language);
        key.put("JntOpgAgrmt", jntOpgAgrmt);
        return new GetByKeyRequestBuilder<JntOpgAgrmtText>(
            servicePath,
            JntOpgAgrmtText.class,
            key,
            "A_JntOpgAgrmtText");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<JntOpgAgrmtText> createJntOpgAgrmtText( @Nonnull final JntOpgAgrmtText jntOpgAgrmtText )
    {
        return new CreateRequestBuilder<JntOpgAgrmtText>(servicePath, jntOpgAgrmtText, "A_JntOpgAgrmtText");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<JntOpgAgrmtText> updateJntOpgAgrmtText( @Nonnull final JntOpgAgrmtText jntOpgAgrmtText )
    {
        return new UpdateRequestBuilder<JntOpgAgrmtText>(servicePath, jntOpgAgrmtText, "A_JntOpgAgrmtText");
    }

    @Override
    @Nonnull
    public DeleteRequestBuilder<JntOpgAgrmtText> deleteJntOpgAgrmtText( @Nonnull final JntOpgAgrmtText jntOpgAgrmtText )
    {
        return new DeleteRequestBuilder<JntOpgAgrmtText>(servicePath, jntOpgAgrmtText, "A_JntOpgAgrmtText");
    }

}
