/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JntVntrPartEquityShr;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVenture;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureDescription;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityGroup;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityType;

import lombok.Getter;

/**
 * <p>
 * The service contains Joint Venture, Description, Equity Group, Equity Type, and Equity Share nodes. The information
 * is sent in the request as a payload, which is used in creating or updating a joint venture, including appropriate
 * checks for restricted values and authorizations. If there are any issues when the joint venture is created or
 * updated, the system displays error messages in the response. The service does not allow deletion of any of joint
 * ventures, descriptions, equity group, equity type or equity share.
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/a9/bbf241929d4abe8e685c877cc26428/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/API_JOINTVENTURE_0001?section=OVERVIEW'>SAP API
 * Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>API_JOINTVENTURE_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Joint Venture Accounting API Integration (SAP_COM_0383)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/3F7'>Joint Venture Accounting (3F7)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_JOINTVENTURE</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultJointVentureService" )
public class DefaultJointVentureService implements ServiceWithNavigableEntities, JointVentureService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link JointVentureService#DEFAULT_SERVICE_PATH} to send the requests.
     * 
     */
    public DefaultJointVentureService()
    {
        servicePath = JointVentureService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultJointVentureService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultJointVentureService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultJointVentureService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<JntVntrPartEquityShr> getAllJntVntrPartEquityShr()
    {
        return new GetAllRequestBuilder<JntVntrPartEquityShr>(
            servicePath,
            JntVntrPartEquityShr.class,
            "A_JntVntrPartEquityShr");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<JntVntrPartEquityShr> countJntVntrPartEquityShr()
    {
        return new CountRequestBuilder<JntVntrPartEquityShr>(
            servicePath,
            JntVntrPartEquityShr.class,
            "A_JntVntrPartEquityShr");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<JntVntrPartEquityShr> getJntVntrPartEquityShrByKey(
        @Nonnull final String companyCode,
        @Nonnull final String jointVenture,
        @Nonnull final String jntOpgAgrmtEquityGrp,
        @Nonnull final String jointVenturePartner )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("CompanyCode", companyCode);
        key.put("JointVenture", jointVenture);
        key.put("JntOpgAgrmtEquityGrp", jntOpgAgrmtEquityGrp);
        key.put("JointVenturePartner", jointVenturePartner);
        return new GetByKeyRequestBuilder<JntVntrPartEquityShr>(
            servicePath,
            JntVntrPartEquityShr.class,
            key,
            "A_JntVntrPartEquityShr");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<JntVntrPartEquityShr> createJntVntrPartEquityShr(
        @Nonnull final JntVntrPartEquityShr jntVntrPartEquityShr )
    {
        return new CreateRequestBuilder<JntVntrPartEquityShr>(
            servicePath,
            jntVntrPartEquityShr,
            "A_JntVntrPartEquityShr");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<JntVntrPartEquityShr> updateJntVntrPartEquityShr(
        @Nonnull final JntVntrPartEquityShr jntVntrPartEquityShr )
    {
        return new UpdateRequestBuilder<JntVntrPartEquityShr>(
            servicePath,
            jntVntrPartEquityShr,
            "A_JntVntrPartEquityShr");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<JointVenture> getAllJointVenture()
    {
        return new GetAllRequestBuilder<JointVenture>(servicePath, JointVenture.class, "A_JointVenture");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<JointVenture> countJointVenture()
    {
        return new CountRequestBuilder<JointVenture>(servicePath, JointVenture.class, "A_JointVenture");
    }

    @Override
    @Nonnull
    public
        GetByKeyRequestBuilder<JointVenture>
        getJointVentureByKey( @Nonnull final String companyCode, @Nonnull final String jointVenture )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("CompanyCode", companyCode);
        key.put("JointVenture", jointVenture);
        return new GetByKeyRequestBuilder<JointVenture>(servicePath, JointVenture.class, key, "A_JointVenture");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<JointVenture> createJointVenture( @Nonnull final JointVenture jointVenture )
    {
        return new CreateRequestBuilder<JointVenture>(servicePath, jointVenture, "A_JointVenture");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<JointVenture> updateJointVenture( @Nonnull final JointVenture jointVenture )
    {
        return new UpdateRequestBuilder<JointVenture>(servicePath, jointVenture, "A_JointVenture");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<JointVentureDescription> getAllJointVentureDescription()
    {
        return new GetAllRequestBuilder<JointVentureDescription>(
            servicePath,
            JointVentureDescription.class,
            "A_JointVentureDescription");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<JointVentureDescription> countJointVentureDescription()
    {
        return new CountRequestBuilder<JointVentureDescription>(
            servicePath,
            JointVentureDescription.class,
            "A_JointVentureDescription");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<JointVentureDescription> getJointVentureDescriptionByKey(
        @Nonnull final String companyCode,
        @Nonnull final String language,
        @Nonnull final String jointVenture )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("CompanyCode", companyCode);
        key.put("Language", language);
        key.put("JointVenture", jointVenture);
        return new GetByKeyRequestBuilder<JointVentureDescription>(
            servicePath,
            JointVentureDescription.class,
            key,
            "A_JointVentureDescription");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<JointVentureDescription> createJointVentureDescription(
        @Nonnull final JointVentureDescription jointVentureDescription )
    {
        return new CreateRequestBuilder<JointVentureDescription>(
            servicePath,
            jointVentureDescription,
            "A_JointVentureDescription");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<JointVentureDescription> updateJointVentureDescription(
        @Nonnull final JointVentureDescription jointVentureDescription )
    {
        return new UpdateRequestBuilder<JointVentureDescription>(
            servicePath,
            jointVentureDescription,
            "A_JointVentureDescription");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<JointVentureEquityGroup> getAllJointVentureEquityGroup()
    {
        return new GetAllRequestBuilder<JointVentureEquityGroup>(
            servicePath,
            JointVentureEquityGroup.class,
            "A_JointVentureEquityGroup");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<JointVentureEquityGroup> countJointVentureEquityGroup()
    {
        return new CountRequestBuilder<JointVentureEquityGroup>(
            servicePath,
            JointVentureEquityGroup.class,
            "A_JointVentureEquityGroup");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<JointVentureEquityGroup> getJointVentureEquityGroupByKey(
        @Nonnull final String companyCode,
        @Nonnull final String jointVenture,
        @Nonnull final String jntOpgAgrmtEquityGrp )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("CompanyCode", companyCode);
        key.put("JointVenture", jointVenture);
        key.put("JntOpgAgrmtEquityGrp", jntOpgAgrmtEquityGrp);
        return new GetByKeyRequestBuilder<JointVentureEquityGroup>(
            servicePath,
            JointVentureEquityGroup.class,
            key,
            "A_JointVentureEquityGroup");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<JointVentureEquityGroup> createJointVentureEquityGroup(
        @Nonnull final JointVentureEquityGroup jointVentureEquityGroup )
    {
        return new CreateRequestBuilder<JointVentureEquityGroup>(
            servicePath,
            jointVentureEquityGroup,
            "A_JointVentureEquityGroup");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<JointVentureEquityGroup> updateJointVentureEquityGroup(
        @Nonnull final JointVentureEquityGroup jointVentureEquityGroup )
    {
        return new UpdateRequestBuilder<JointVentureEquityGroup>(
            servicePath,
            jointVentureEquityGroup,
            "A_JointVentureEquityGroup");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<JointVentureEquityType> getAllJointVentureEquityType()
    {
        return new GetAllRequestBuilder<JointVentureEquityType>(
            servicePath,
            JointVentureEquityType.class,
            "A_JointVentureEquityType");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<JointVentureEquityType> countJointVentureEquityType()
    {
        return new CountRequestBuilder<JointVentureEquityType>(
            servicePath,
            JointVentureEquityType.class,
            "A_JointVentureEquityType");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<JointVentureEquityType> getJointVentureEquityTypeByKey(
        @Nonnull final String companyCode,
        @Nonnull final String jointVenture,
        @Nonnull final String jointVentureEquityType,
        @Nonnull final String jointVentureEquityFundingDate )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("CompanyCode", companyCode);
        key.put("JointVenture", jointVenture);
        key.put("JointVentureEquityType", jointVentureEquityType);
        key.put("JointVentureEquityFundingDate", jointVentureEquityFundingDate);
        return new GetByKeyRequestBuilder<JointVentureEquityType>(
            servicePath,
            JointVentureEquityType.class,
            key,
            "A_JointVentureEquityType");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<JointVentureEquityType> createJointVentureEquityType(
        @Nonnull final JointVentureEquityType jointVentureEquityType )
    {
        return new CreateRequestBuilder<JointVentureEquityType>(
            servicePath,
            jointVentureEquityType,
            "A_JointVentureEquityType");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<JointVentureEquityType> updateJointVentureEquityType(
        @Nonnull final JointVentureEquityType jointVentureEquityType )
    {
        return new UpdateRequestBuilder<JointVentureEquityType>(
            servicePath,
            jointVentureEquityType,
            "A_JointVentureEquityType");
    }

}
