/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.DeleteRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.maintainflexibleconstraintsforpredictivematerialandresourceplanning.ConstraintProduct;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.maintainflexibleconstraintsforpredictivematerialandresourceplanning.PMRPFlexibleConstraint;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.maintainflexibleconstraintsforpredictivematerialandresourceplanning.Period;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.maintainflexibleconstraintsforpredictivematerialandresourceplanning.Text;

import lombok.Getter;

/**
 * <p>
 * 
 * Constraints are things that limit your production process in some way. For example, if your supplier has a shortage
 * of a key component in your product, you are limited in the number of products you can create. The constraint can be
 * long-term or short-term and could be caused by any number of factors, including, supply shortages, or reduced
 * staffing.
 * 
 * This service enables you to define your own constraints for many different types of material. This means that no
 * matter the reason for the constraint, you can flexibly define which materials are affected, how they are affected,
 * and for how long. The defined constraints are then considered in forecasting simulations..
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/fb/d33819088749ee80593f09d504ae2e/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/PMRPFLEXIBLECONSTRAINT_0001?section=OVERVIEW'>SAP
 * API Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>PMRPFLEXIBLECONSTRAINT_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Predictive MRP Masterdata Integration (SAP_COM_0739)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/4B5'>Predictive Material and Resource Planning (4B5)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509, OAuth2</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_PMRPFLEXIBLECONSTRAINT</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>PMRPFlexibleConstraint</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultMaintainFlexibleConstraintsForPredictiveMaterialAndResourcePlanningService" )
public class DefaultMaintainFlexibleConstraintsForPredictiveMaterialAndResourcePlanningService
    implements
    ServiceWithNavigableEntities,
    MaintainFlexibleConstraintsForPredictiveMaterialAndResourcePlanningService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using
     * {@link MaintainFlexibleConstraintsForPredictiveMaterialAndResourcePlanningService#DEFAULT_SERVICE_PATH} to send
     * the requests.
     * 
     */
    public DefaultMaintainFlexibleConstraintsForPredictiveMaterialAndResourcePlanningService()
    {
        servicePath = MaintainFlexibleConstraintsForPredictiveMaterialAndResourcePlanningService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultMaintainFlexibleConstraintsForPredictiveMaterialAndResourcePlanningService(
        @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultMaintainFlexibleConstraintsForPredictiveMaterialAndResourcePlanningService withServicePath(
        @Nonnull final String servicePath )
    {
        return new DefaultMaintainFlexibleConstraintsForPredictiveMaterialAndResourcePlanningService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<ConstraintProduct> getAllConstraintProduct()
    {
        return new GetAllRequestBuilder<ConstraintProduct>(servicePath, ConstraintProduct.class, "ConstraintProduct");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<ConstraintProduct> countConstraintProduct()
    {
        return new CountRequestBuilder<ConstraintProduct>(servicePath, ConstraintProduct.class, "ConstraintProduct");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<ConstraintProduct> getConstraintProductByKey(
        @Nonnull final String pMRPFlexibleConstraint,
        @Nonnull final String pMRPConstraintProductItem )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("PMRPFlexibleConstraint", pMRPFlexibleConstraint);
        key.put("PMRPConstraintProductItem", pMRPConstraintProductItem);
        return new GetByKeyRequestBuilder<ConstraintProduct>(
            servicePath,
            ConstraintProduct.class,
            key,
            "ConstraintProduct");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<ConstraintProduct> createConstraintProduct(
        @Nonnull final ConstraintProduct constraintProduct )
    {
        return new CreateRequestBuilder<ConstraintProduct>(servicePath, constraintProduct, "ConstraintProduct");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<Period> getAllPeriod()
    {
        return new GetAllRequestBuilder<Period>(servicePath, Period.class, "Period");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<Period> countPeriod()
    {
        return new CountRequestBuilder<Period>(servicePath, Period.class, "Period");
    }

    @Override
    @Nonnull
    public
        GetByKeyRequestBuilder<Period>
        getPeriodByKey( @Nonnull final String pMRPFlexibleConstraint, @Nonnull final Integer bucketInternalID )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("PMRPFlexibleConstraint", pMRPFlexibleConstraint);
        key.put("BucketInternalID", bucketInternalID);
        return new GetByKeyRequestBuilder<Period>(servicePath, Period.class, key, "Period");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<Period> createPeriod( @Nonnull final Period period )
    {
        return new CreateRequestBuilder<Period>(servicePath, period, "Period");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<Period> updatePeriod( @Nonnull final Period period )
    {
        return new UpdateRequestBuilder<Period>(servicePath, period, "Period");
    }

    @Override
    @Nonnull
    public DeleteRequestBuilder<Period> deletePeriod( @Nonnull final Period period )
    {
        return new DeleteRequestBuilder<Period>(servicePath, period, "Period");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<PMRPFlexibleConstraint> getAllPMRPFlexibleConstraint()
    {
        return new GetAllRequestBuilder<PMRPFlexibleConstraint>(
            servicePath,
            PMRPFlexibleConstraint.class,
            "PMRPFlexibleConstraint");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<PMRPFlexibleConstraint> countPMRPFlexibleConstraint()
    {
        return new CountRequestBuilder<PMRPFlexibleConstraint>(
            servicePath,
            PMRPFlexibleConstraint.class,
            "PMRPFlexibleConstraint");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<PMRPFlexibleConstraint> getPMRPFlexibleConstraintByKey(
        @Nonnull final String pMRPFlexibleConstraint )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("PMRPFlexibleConstraint", pMRPFlexibleConstraint);
        return new GetByKeyRequestBuilder<PMRPFlexibleConstraint>(
            servicePath,
            PMRPFlexibleConstraint.class,
            key,
            "PMRPFlexibleConstraint");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<PMRPFlexibleConstraint> createPMRPFlexibleConstraint(
        @Nonnull final PMRPFlexibleConstraint pMRPFlexibleConstraint )
    {
        return new CreateRequestBuilder<PMRPFlexibleConstraint>(
            servicePath,
            pMRPFlexibleConstraint,
            "PMRPFlexibleConstraint");
    }

    @Override
    @Nonnull
    public DeleteRequestBuilder<PMRPFlexibleConstraint> deletePMRPFlexibleConstraint(
        @Nonnull final PMRPFlexibleConstraint pMRPFlexibleConstraint )
    {
        return new DeleteRequestBuilder<PMRPFlexibleConstraint>(
            servicePath,
            pMRPFlexibleConstraint,
            "PMRPFlexibleConstraint");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<Text> getAllText()
    {
        return new GetAllRequestBuilder<Text>(servicePath, Text.class, "Text");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<Text> countText()
    {
        return new CountRequestBuilder<Text>(servicePath, Text.class, "Text");
    }

    @Override
    @Nonnull
    public
        GetByKeyRequestBuilder<Text>
        getTextByKey( @Nonnull final String pMRPFlexibleConstraint, @Nonnull final String language )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("PMRPFlexibleConstraint", pMRPFlexibleConstraint);
        key.put("Language", language);
        return new GetByKeyRequestBuilder<Text>(servicePath, Text.class, key, "Text");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<Text> createText( @Nonnull final Text text )
    {
        return new CreateRequestBuilder<Text>(servicePath, text, "Text");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<Text> updateText( @Nonnull final Text text )
    {
        return new UpdateRequestBuilder<Text>(servicePath, text, "Text");
    }

}
