/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.maintenancetasklist.MaintenanceTaskList;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.maintenancetasklist.MaintenanceTaskListLongText;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.maintenancetasklist.MaintenanceTaskListOpLongText;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.maintenancetasklist.MaintenanceTaskListOpMat;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.maintenancetasklist.MaintenanceTaskListOpReltnshp;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.maintenancetasklist.MaintenanceTaskListOperation;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.maintenancetasklist.MaintenanceTaskListStrtgyPckg;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.maintenancetasklist.MaintenanceTaskListSubOp;

import lombok.Getter;

/**
 * <p>
 * 
 * The service describes the sequence of individual maintenance activities that must be performed repeatedly within a
 * company. There are three types of task list. The task list comprises equipment task list, functional location task
 * list, and general maintenance task list. The payload used to create a task list through this API is sent in JSON
 * format as a request object. The long text of the task list is included within the payload as a nested object. .
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/21/d658b2c6e7444f87d0cef5cdd40b7f/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/MAINTENANCETASKLIST_0001?section=OVERVIEW'>SAP
 * API Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>MAINTENANCETASKLIST_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Asset Management Master Data Integration (SAP_COM_0395)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/3D2'>Service Order Management and Monitoring (3D2)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/3MO'>Service Contract Management (3MO)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/3XK'>In-House Repair (3XK)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/49X'>Service Order Processing with SAP Field Service Management
 * (49X)</a>, <a href='https://rapid.sap.com/bp/#/scopeitems/4GA'>Service Quotation (4GA)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/4HH'>Reactive Maintenance (4HH)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/4HI'>Proactive Maintenance (4HI)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/4VT'>Improvement Maintenance (4VT)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/4WM'>Operational and Overhead Maintenance (4WM)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BH1'>Corrective Maintenance (BH1)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BH2'>Emergency Maintenance (BH2)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BJ2'>Preventive Maintenance (BJ2)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509, OAuth2</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_MAINTENANCETASKLIST</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>MaintenanceTaskList</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultMaintenanceTasklistService" )
public class DefaultMaintenanceTasklistService implements ServiceWithNavigableEntities, MaintenanceTasklistService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link MaintenanceTasklistService#DEFAULT_SERVICE_PATH} to send the requests.
     * 
     */
    public DefaultMaintenanceTasklistService()
    {
        servicePath = MaintenanceTasklistService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultMaintenanceTasklistService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultMaintenanceTasklistService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultMaintenanceTasklistService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<MaintenanceTaskList> getAllMaintenanceTaskList()
    {
        return new GetAllRequestBuilder<MaintenanceTaskList>(
            servicePath,
            MaintenanceTaskList.class,
            "MaintenanceTaskList");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<MaintenanceTaskList> countMaintenanceTaskList()
    {
        return new CountRequestBuilder<MaintenanceTaskList>(
            servicePath,
            MaintenanceTaskList.class,
            "MaintenanceTaskList");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<MaintenanceTaskList> getMaintenanceTaskListByKey(
        @Nonnull final String taskListType,
        @Nonnull final String taskListGroup,
        @Nonnull final String taskListGroupCounter,
        @Nonnull final String taskListVersionCounter )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("TaskListType", taskListType);
        key.put("TaskListGroup", taskListGroup);
        key.put("TaskListGroupCounter", taskListGroupCounter);
        key.put("TaskListVersionCounter", taskListVersionCounter);
        return new GetByKeyRequestBuilder<MaintenanceTaskList>(
            servicePath,
            MaintenanceTaskList.class,
            key,
            "MaintenanceTaskList");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<MaintenanceTaskList> createMaintenanceTaskList(
        @Nonnull final MaintenanceTaskList maintenanceTaskList )
    {
        return new CreateRequestBuilder<MaintenanceTaskList>(servicePath, maintenanceTaskList, "MaintenanceTaskList");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<MaintenanceTaskList> updateMaintenanceTaskList(
        @Nonnull final MaintenanceTaskList maintenanceTaskList )
    {
        return new UpdateRequestBuilder<MaintenanceTaskList>(servicePath, maintenanceTaskList, "MaintenanceTaskList");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<MaintenanceTaskListLongText> getAllMaintenanceTaskListLongText()
    {
        return new GetAllRequestBuilder<MaintenanceTaskListLongText>(
            servicePath,
            MaintenanceTaskListLongText.class,
            "MaintenanceTaskListLongText");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<MaintenanceTaskListLongText> countMaintenanceTaskListLongText()
    {
        return new CountRequestBuilder<MaintenanceTaskListLongText>(
            servicePath,
            MaintenanceTaskListLongText.class,
            "MaintenanceTaskListLongText");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<MaintenanceTaskListLongText> getMaintenanceTaskListLongTextByKey(
        @Nonnull final String taskListType,
        @Nonnull final String taskListGroup,
        @Nonnull final String taskListGroupCounter,
        @Nonnull final String taskListVersionCounter )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("TaskListType", taskListType);
        key.put("TaskListGroup", taskListGroup);
        key.put("TaskListGroupCounter", taskListGroupCounter);
        key.put("TaskListVersionCounter", taskListVersionCounter);
        return new GetByKeyRequestBuilder<MaintenanceTaskListLongText>(
            servicePath,
            MaintenanceTaskListLongText.class,
            key,
            "MaintenanceTaskListLongText");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<MaintenanceTaskListLongText> createMaintenanceTaskListLongText(
        @Nonnull final MaintenanceTaskListLongText maintenanceTaskListLongText )
    {
        return new CreateRequestBuilder<MaintenanceTaskListLongText>(
            servicePath,
            maintenanceTaskListLongText,
            "MaintenanceTaskListLongText");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<MaintenanceTaskListLongText> updateMaintenanceTaskListLongText(
        @Nonnull final MaintenanceTaskListLongText maintenanceTaskListLongText )
    {
        return new UpdateRequestBuilder<MaintenanceTaskListLongText>(
            servicePath,
            maintenanceTaskListLongText,
            "MaintenanceTaskListLongText");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<MaintenanceTaskListOperation> getAllMaintenanceTaskListOperation()
    {
        return new GetAllRequestBuilder<MaintenanceTaskListOperation>(
            servicePath,
            MaintenanceTaskListOperation.class,
            "MaintenanceTaskListOperation");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<MaintenanceTaskListOperation> countMaintenanceTaskListOperation()
    {
        return new CountRequestBuilder<MaintenanceTaskListOperation>(
            servicePath,
            MaintenanceTaskListOperation.class,
            "MaintenanceTaskListOperation");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<MaintenanceTaskListOperation> getMaintenanceTaskListOperationByKey(
        @Nonnull final String taskListType,
        @Nonnull final String taskListGroup,
        @Nonnull final String taskListGroupCounter,
        @Nonnull final String taskListSequence,
        @Nonnull final String taskListOperationInternalId,
        @Nonnull final String taskListOpBOMItmIntVersCounter )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("TaskListType", taskListType);
        key.put("TaskListGroup", taskListGroup);
        key.put("TaskListGroupCounter", taskListGroupCounter);
        key.put("TaskListSequence", taskListSequence);
        key.put("TaskListOperationInternalId", taskListOperationInternalId);
        key.put("TaskListOpBOMItmIntVersCounter", taskListOpBOMItmIntVersCounter);
        return new GetByKeyRequestBuilder<MaintenanceTaskListOperation>(
            servicePath,
            MaintenanceTaskListOperation.class,
            key,
            "MaintenanceTaskListOperation");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<MaintenanceTaskListOperation> createMaintenanceTaskListOperation(
        @Nonnull final MaintenanceTaskListOperation maintenanceTaskListOperation )
    {
        return new CreateRequestBuilder<MaintenanceTaskListOperation>(
            servicePath,
            maintenanceTaskListOperation,
            "MaintenanceTaskListOperation");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<MaintenanceTaskListOperation> updateMaintenanceTaskListOperation(
        @Nonnull final MaintenanceTaskListOperation maintenanceTaskListOperation )
    {
        return new UpdateRequestBuilder<MaintenanceTaskListOperation>(
            servicePath,
            maintenanceTaskListOperation,
            "MaintenanceTaskListOperation");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<MaintenanceTaskListOpLongText> getAllMaintenanceTaskListOpLongText()
    {
        return new GetAllRequestBuilder<MaintenanceTaskListOpLongText>(
            servicePath,
            MaintenanceTaskListOpLongText.class,
            "MaintenanceTaskListOpLongText");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<MaintenanceTaskListOpLongText> countMaintenanceTaskListOpLongText()
    {
        return new CountRequestBuilder<MaintenanceTaskListOpLongText>(
            servicePath,
            MaintenanceTaskListOpLongText.class,
            "MaintenanceTaskListOpLongText");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<MaintenanceTaskListOpLongText> getMaintenanceTaskListOpLongTextByKey(
        @Nonnull final String taskListType,
        @Nonnull final String taskListGroup,
        @Nonnull final String taskListGroupCounter,
        @Nonnull final String taskListSequence,
        @Nonnull final String taskListOperationInternalId,
        @Nonnull final String taskListOpBOMItmIntVersCounter )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("TaskListType", taskListType);
        key.put("TaskListGroup", taskListGroup);
        key.put("TaskListGroupCounter", taskListGroupCounter);
        key.put("TaskListSequence", taskListSequence);
        key.put("TaskListOperationInternalId", taskListOperationInternalId);
        key.put("TaskListOpBOMItmIntVersCounter", taskListOpBOMItmIntVersCounter);
        return new GetByKeyRequestBuilder<MaintenanceTaskListOpLongText>(
            servicePath,
            MaintenanceTaskListOpLongText.class,
            key,
            "MaintenanceTaskListOpLongText");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<MaintenanceTaskListOpLongText> createMaintenanceTaskListOpLongText(
        @Nonnull final MaintenanceTaskListOpLongText maintenanceTaskListOpLongText )
    {
        return new CreateRequestBuilder<MaintenanceTaskListOpLongText>(
            servicePath,
            maintenanceTaskListOpLongText,
            "MaintenanceTaskListOpLongText");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<MaintenanceTaskListOpLongText> updateMaintenanceTaskListOpLongText(
        @Nonnull final MaintenanceTaskListOpLongText maintenanceTaskListOpLongText )
    {
        return new UpdateRequestBuilder<MaintenanceTaskListOpLongText>(
            servicePath,
            maintenanceTaskListOpLongText,
            "MaintenanceTaskListOpLongText");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<MaintenanceTaskListOpMat> getAllMaintenanceTaskListOpMat()
    {
        return new GetAllRequestBuilder<MaintenanceTaskListOpMat>(
            servicePath,
            MaintenanceTaskListOpMat.class,
            "MaintenanceTaskListOpMat");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<MaintenanceTaskListOpMat> countMaintenanceTaskListOpMat()
    {
        return new CountRequestBuilder<MaintenanceTaskListOpMat>(
            servicePath,
            MaintenanceTaskListOpMat.class,
            "MaintenanceTaskListOpMat");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<MaintenanceTaskListOpMat> getMaintenanceTaskListOpMatByKey(
        @Nonnull final String taskListType,
        @Nonnull final String taskListGroup,
        @Nonnull final String taskListGroupCounter,
        @Nonnull final String taskListSequence,
        @Nonnull final String taskListOperationInternalId,
        @Nonnull final String taskListOpBOMItmInternalID,
        @Nonnull final String taskListOpBOMItmIntVersCounter )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("TaskListType", taskListType);
        key.put("TaskListGroup", taskListGroup);
        key.put("TaskListGroupCounter", taskListGroupCounter);
        key.put("TaskListSequence", taskListSequence);
        key.put("TaskListOperationInternalId", taskListOperationInternalId);
        key.put("TaskListOpBOMItmInternalID", taskListOpBOMItmInternalID);
        key.put("TaskListOpBOMItmIntVersCounter", taskListOpBOMItmIntVersCounter);
        return new GetByKeyRequestBuilder<MaintenanceTaskListOpMat>(
            servicePath,
            MaintenanceTaskListOpMat.class,
            key,
            "MaintenanceTaskListOpMat");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<MaintenanceTaskListOpMat> createMaintenanceTaskListOpMat(
        @Nonnull final MaintenanceTaskListOpMat maintenanceTaskListOpMat )
    {
        return new CreateRequestBuilder<MaintenanceTaskListOpMat>(
            servicePath,
            maintenanceTaskListOpMat,
            "MaintenanceTaskListOpMat");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<MaintenanceTaskListOpMat> updateMaintenanceTaskListOpMat(
        @Nonnull final MaintenanceTaskListOpMat maintenanceTaskListOpMat )
    {
        return new UpdateRequestBuilder<MaintenanceTaskListOpMat>(
            servicePath,
            maintenanceTaskListOpMat,
            "MaintenanceTaskListOpMat");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<MaintenanceTaskListOpReltnshp> getAllMaintenanceTaskListOpReltnshp()
    {
        return new GetAllRequestBuilder<MaintenanceTaskListOpReltnshp>(
            servicePath,
            MaintenanceTaskListOpReltnshp.class,
            "MaintenanceTaskListOpReltnshp");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<MaintenanceTaskListOpReltnshp> countMaintenanceTaskListOpReltnshp()
    {
        return new CountRequestBuilder<MaintenanceTaskListOpReltnshp>(
            servicePath,
            MaintenanceTaskListOpReltnshp.class,
            "MaintenanceTaskListOpReltnshp");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<MaintenanceTaskListOpReltnshp> getMaintenanceTaskListOpReltnshpByKey(
        @Nonnull final String taskListType,
        @Nonnull final String taskListGroup,
        @Nonnull final String taskListGroupCounter,
        @Nonnull final String taskListSequence,
        @Nonnull final String taskListOperationInternalId,
        @Nonnull final String taskListOpBOMItmIntVersCounter )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("TaskListType", taskListType);
        key.put("TaskListGroup", taskListGroup);
        key.put("TaskListGroupCounter", taskListGroupCounter);
        key.put("TaskListSequence", taskListSequence);
        key.put("TaskListOperationInternalId", taskListOperationInternalId);
        key.put("TaskListOpBOMItmIntVersCounter", taskListOpBOMItmIntVersCounter);
        return new GetByKeyRequestBuilder<MaintenanceTaskListOpReltnshp>(
            servicePath,
            MaintenanceTaskListOpReltnshp.class,
            key,
            "MaintenanceTaskListOpReltnshp");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<MaintenanceTaskListOpReltnshp> createMaintenanceTaskListOpReltnshp(
        @Nonnull final MaintenanceTaskListOpReltnshp maintenanceTaskListOpReltnshp )
    {
        return new CreateRequestBuilder<MaintenanceTaskListOpReltnshp>(
            servicePath,
            maintenanceTaskListOpReltnshp,
            "MaintenanceTaskListOpReltnshp");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<MaintenanceTaskListOpReltnshp> updateMaintenanceTaskListOpReltnshp(
        @Nonnull final MaintenanceTaskListOpReltnshp maintenanceTaskListOpReltnshp )
    {
        return new UpdateRequestBuilder<MaintenanceTaskListOpReltnshp>(
            servicePath,
            maintenanceTaskListOpReltnshp,
            "MaintenanceTaskListOpReltnshp");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<MaintenanceTaskListStrtgyPckg> getAllMaintenanceTaskListStrtgyPckg()
    {
        return new GetAllRequestBuilder<MaintenanceTaskListStrtgyPckg>(
            servicePath,
            MaintenanceTaskListStrtgyPckg.class,
            "MaintenanceTaskListStrtgyPckg");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<MaintenanceTaskListStrtgyPckg> countMaintenanceTaskListStrtgyPckg()
    {
        return new CountRequestBuilder<MaintenanceTaskListStrtgyPckg>(
            servicePath,
            MaintenanceTaskListStrtgyPckg.class,
            "MaintenanceTaskListStrtgyPckg");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<MaintenanceTaskListStrtgyPckg> getMaintenanceTaskListStrtgyPckgByKey(
        @Nonnull final String taskListType,
        @Nonnull final String taskListGroup,
        @Nonnull final String taskListGroupCounter,
        @Nonnull final String taskListSequence,
        @Nonnull final String taskListOperationInternalId,
        @Nonnull final String maintenancePackage,
        @Nonnull final String maintPckgTskListOpAllocIntNmbr )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("TaskListType", taskListType);
        key.put("TaskListGroup", taskListGroup);
        key.put("TaskListGroupCounter", taskListGroupCounter);
        key.put("TaskListSequence", taskListSequence);
        key.put("TaskListOperationInternalId", taskListOperationInternalId);
        key.put("MaintenancePackage", maintenancePackage);
        key.put("MaintPckgTskListOpAllocIntNmbr", maintPckgTskListOpAllocIntNmbr);
        return new GetByKeyRequestBuilder<MaintenanceTaskListStrtgyPckg>(
            servicePath,
            MaintenanceTaskListStrtgyPckg.class,
            key,
            "MaintenanceTaskListStrtgyPckg");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<MaintenanceTaskListStrtgyPckg> createMaintenanceTaskListStrtgyPckg(
        @Nonnull final MaintenanceTaskListStrtgyPckg maintenanceTaskListStrtgyPckg )
    {
        return new CreateRequestBuilder<MaintenanceTaskListStrtgyPckg>(
            servicePath,
            maintenanceTaskListStrtgyPckg,
            "MaintenanceTaskListStrtgyPckg");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<MaintenanceTaskListStrtgyPckg> updateMaintenanceTaskListStrtgyPckg(
        @Nonnull final MaintenanceTaskListStrtgyPckg maintenanceTaskListStrtgyPckg )
    {
        return new UpdateRequestBuilder<MaintenanceTaskListStrtgyPckg>(
            servicePath,
            maintenanceTaskListStrtgyPckg,
            "MaintenanceTaskListStrtgyPckg");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<MaintenanceTaskListSubOp> getAllMaintenanceTaskListSubOp()
    {
        return new GetAllRequestBuilder<MaintenanceTaskListSubOp>(
            servicePath,
            MaintenanceTaskListSubOp.class,
            "MaintenanceTaskListSubOp");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<MaintenanceTaskListSubOp> countMaintenanceTaskListSubOp()
    {
        return new CountRequestBuilder<MaintenanceTaskListSubOp>(
            servicePath,
            MaintenanceTaskListSubOp.class,
            "MaintenanceTaskListSubOp");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<MaintenanceTaskListSubOp> getMaintenanceTaskListSubOpByKey(
        @Nonnull final String taskListType,
        @Nonnull final String taskListGroup,
        @Nonnull final String taskListGroupCounter,
        @Nonnull final String taskListSequence,
        @Nonnull final String taskListOperationInternalId,
        @Nonnull final String taskListOpBOMItmIntVersCounter )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("TaskListType", taskListType);
        key.put("TaskListGroup", taskListGroup);
        key.put("TaskListGroupCounter", taskListGroupCounter);
        key.put("TaskListSequence", taskListSequence);
        key.put("TaskListOperationInternalId", taskListOperationInternalId);
        key.put("TaskListOpBOMItmIntVersCounter", taskListOpBOMItmIntVersCounter);
        return new GetByKeyRequestBuilder<MaintenanceTaskListSubOp>(
            servicePath,
            MaintenanceTaskListSubOp.class,
            key,
            "MaintenanceTaskListSubOp");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<MaintenanceTaskListSubOp> createMaintenanceTaskListSubOp(
        @Nonnull final MaintenanceTaskListSubOp maintenanceTaskListSubOp )
    {
        return new CreateRequestBuilder<MaintenanceTaskListSubOp>(
            servicePath,
            maintenanceTaskListSubOp,
            "MaintenanceTaskListSubOp");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<MaintenanceTaskListSubOp> updateMaintenanceTaskListSubOp(
        @Nonnull final MaintenanceTaskListSubOp maintenanceTaskListSubOp )
    {
        return new UpdateRequestBuilder<MaintenanceTaskListSubOp>(
            servicePath,
            maintenanceTaskListSubOp,
            "MaintenanceTaskListSubOp");
    }

}
