/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.time.LocalDate;
import java.time.LocalTime;
import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.FailedMeasurementReading;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MeasurementDocument;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MsmtDocumentLongText;

import lombok.Getter;

/**
 * <p>
 * 
 * The data transferred to the system after a measurement has been taken at a measuring point or a counter is described
 * in the SAP system as a measurement document.
 * 
 * The payload used to create a measurement document through this API is sent in JSON format as a request object. The
 * long text for the measurement document is included within as a nested object. .
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/6a/fa93607aa74b1bb0aebdbe2867c8f9/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/MEASUREMENTDOCUMENT_0001?section=OVERVIEW'>SAP
 * API Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>MEASUREMENTDOCUMENT_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Plant Maintenance Execution Integration (SAP_COM_0398)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/3D2'>Service Order Management and Monitoring (3D2)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/3MO'>Service Contract Management (3MO)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/3XK'>In-House Repair (3XK)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/49X'>Service Order Processing with SAP Field Service Management
 * (49X)</a>, <a href='https://rapid.sap.com/bp/#/scopeitems/4GA'>Service Quotation (4GA)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/4HH'>Reactive Maintenance (4HH)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/4HI'>Proactive Maintenance (4HI)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/4VT'>Improvement Maintenance (4VT)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/4WM'>Operational and Overhead Maintenance (4WM)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BH1'>Corrective Maintenance (BH1)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BH2'>Emergency Maintenance (BH2)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BJ2'>Preventive Maintenance (BJ2)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509, OAuth2</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_MEASUREMENTDOCUMENT</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>MeasurementDocument</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultMeasurementDocumentService" )
public class DefaultMeasurementDocumentService implements ServiceWithNavigableEntities, MeasurementDocumentService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link MeasurementDocumentService#DEFAULT_SERVICE_PATH} to send the requests.
     * 
     */
    public DefaultMeasurementDocumentService()
    {
        servicePath = MeasurementDocumentService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultMeasurementDocumentService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultMeasurementDocumentService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultMeasurementDocumentService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<FailedMeasurementReading> getAllFailedMeasurementReading()
    {
        return new GetAllRequestBuilder<FailedMeasurementReading>(
            servicePath,
            FailedMeasurementReading.class,
            "FailedMeasurementReading");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<FailedMeasurementReading> countFailedMeasurementReading()
    {
        return new CountRequestBuilder<FailedMeasurementReading>(
            servicePath,
            FailedMeasurementReading.class,
            "FailedMeasurementReading");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<FailedMeasurementReading> getFailedMeasurementReadingByKey(
        @Nonnull final LocalDate msmtRdngDate,
        @Nonnull final LocalTime msmtRdngTime,
        @Nonnull final String msmtRdngSourceMeasurementDoc,
        @Nonnull final String measuringPoint )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("MsmtRdngDate", msmtRdngDate);
        key.put("MsmtRdngTime", msmtRdngTime);
        key.put("MsmtRdngSourceMeasurementDoc", msmtRdngSourceMeasurementDoc);
        key.put("MeasuringPoint", measuringPoint);
        return new GetByKeyRequestBuilder<FailedMeasurementReading>(
            servicePath,
            FailedMeasurementReading.class,
            key,
            "FailedMeasurementReading");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<FailedMeasurementReading> createFailedMeasurementReading(
        @Nonnull final FailedMeasurementReading failedMeasurementReading )
    {
        return new CreateRequestBuilder<FailedMeasurementReading>(
            servicePath,
            failedMeasurementReading,
            "FailedMeasurementReading");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<MeasurementDocument> getAllMeasurementDocument()
    {
        return new GetAllRequestBuilder<MeasurementDocument>(
            servicePath,
            MeasurementDocument.class,
            "MeasurementDocument");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<MeasurementDocument> countMeasurementDocument()
    {
        return new CountRequestBuilder<MeasurementDocument>(
            servicePath,
            MeasurementDocument.class,
            "MeasurementDocument");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<MeasurementDocument> getMeasurementDocumentByKey(
        @Nonnull final String measurementDocument )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("MeasurementDocument", measurementDocument);
        return new GetByKeyRequestBuilder<MeasurementDocument>(
            servicePath,
            MeasurementDocument.class,
            key,
            "MeasurementDocument");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<MeasurementDocument> createMeasurementDocument(
        @Nonnull final MeasurementDocument measurementDocument )
    {
        return new CreateRequestBuilder<MeasurementDocument>(servicePath, measurementDocument, "MeasurementDocument");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<MeasurementDocument> updateMeasurementDocument(
        @Nonnull final MeasurementDocument measurementDocument )
    {
        return new UpdateRequestBuilder<MeasurementDocument>(servicePath, measurementDocument, "MeasurementDocument");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<MsmtDocumentLongText> getAllMsmtDocumentLongText()
    {
        return new GetAllRequestBuilder<MsmtDocumentLongText>(
            servicePath,
            MsmtDocumentLongText.class,
            "MsmtDocumentLongText");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<MsmtDocumentLongText> countMsmtDocumentLongText()
    {
        return new CountRequestBuilder<MsmtDocumentLongText>(
            servicePath,
            MsmtDocumentLongText.class,
            "MsmtDocumentLongText");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<MsmtDocumentLongText> getMsmtDocumentLongTextByKey(
        @Nonnull final String measurementDocument )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("MeasurementDocument", measurementDocument);
        return new GetByKeyRequestBuilder<MsmtDocumentLongText>(
            servicePath,
            MsmtDocumentLongText.class,
            key,
            "MsmtDocumentLongText");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<MsmtDocumentLongText> createMsmtDocumentLongText(
        @Nonnull final MsmtDocumentLongText msmtDocumentLongText )
    {
        return new CreateRequestBuilder<MsmtDocumentLongText>(
            servicePath,
            msmtDocumentLongText,
            "MsmtDocumentLongText");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<MsmtDocumentLongText> updateMsmtDocumentLongText(
        @Nonnull final MsmtDocumentLongText msmtDocumentLongText )
    {
        return new UpdateRequestBuilder<MsmtDocumentLongText>(
            servicePath,
            msmtDocumentLongText,
            "MsmtDocumentLongText");
    }

}
