/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measuringpoint.MeasuringPoint;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measuringpoint.MeasuringPointLongText;

import lombok.Getter;

/**
 * <p>
 * 
 * In Plant Maintenance, measuring points are located on technical objects such as pieces of equipment or at functional
 * locations. The payload used to create a measuring point through this API is sent in JSON format as a request object.
 * The long text for the measuring point is included within as a nested object. .
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/8c/dfef769b2b4f7195a5f296982e2fe6/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/MEASURINGPOINT_0001?section=OVERVIEW'>SAP API
 * Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>MEASURINGPOINT_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Asset Management Master Data Integration (SAP_COM_0395)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/3D2'>Service Order Management and Monitoring (3D2)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/3MO'>Service Contract Management (3MO)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/3XK'>In-House Repair (3XK)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/49X'>Service Order Processing with SAP Field Service Management
 * (49X)</a>, <a href='https://rapid.sap.com/bp/#/scopeitems/4GA'>Service Quotation (4GA)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/4HH'>Reactive Maintenance (4HH)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/4HI'>Proactive Maintenance (4HI)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/4VT'>Improvement Maintenance (4VT)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/4WM'>Operational and Overhead Maintenance (4WM)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BH1'>Corrective Maintenance (BH1)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BH2'>Emergency Maintenance (BH2)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BJ2'>Preventive Maintenance (BJ2)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509, OAuth2</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_MEASURINGPOINT</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>MeasuringPoint</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultMeasuringPointService" )
public class DefaultMeasuringPointService implements ServiceWithNavigableEntities, MeasuringPointService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link MeasuringPointService#DEFAULT_SERVICE_PATH} to send the requests.
     * 
     */
    public DefaultMeasuringPointService()
    {
        servicePath = MeasuringPointService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultMeasuringPointService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultMeasuringPointService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultMeasuringPointService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<MeasuringPoint> getAllMeasuringPoint()
    {
        return new GetAllRequestBuilder<MeasuringPoint>(servicePath, MeasuringPoint.class, "MeasuringPoint");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<MeasuringPoint> countMeasuringPoint()
    {
        return new CountRequestBuilder<MeasuringPoint>(servicePath, MeasuringPoint.class, "MeasuringPoint");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<MeasuringPoint> getMeasuringPointByKey( @Nonnull final String measuringPoint )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("MeasuringPoint", measuringPoint);
        return new GetByKeyRequestBuilder<MeasuringPoint>(servicePath, MeasuringPoint.class, key, "MeasuringPoint");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<MeasuringPoint> createMeasuringPoint( @Nonnull final MeasuringPoint measuringPoint )
    {
        return new CreateRequestBuilder<MeasuringPoint>(servicePath, measuringPoint, "MeasuringPoint");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<MeasuringPoint> updateMeasuringPoint( @Nonnull final MeasuringPoint measuringPoint )
    {
        return new UpdateRequestBuilder<MeasuringPoint>(servicePath, measuringPoint, "MeasuringPoint");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<MeasuringPointLongText> getAllMeasuringPointLongText()
    {
        return new GetAllRequestBuilder<MeasuringPointLongText>(
            servicePath,
            MeasuringPointLongText.class,
            "MeasuringPointLongText");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<MeasuringPointLongText> countMeasuringPointLongText()
    {
        return new CountRequestBuilder<MeasuringPointLongText>(
            servicePath,
            MeasuringPointLongText.class,
            "MeasuringPointLongText");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<MeasuringPointLongText> getMeasuringPointLongTextByKey(
        @Nonnull final String measuringPoint )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("MeasuringPoint", measuringPoint);
        return new GetByKeyRequestBuilder<MeasuringPointLongText>(
            servicePath,
            MeasuringPointLongText.class,
            key,
            "MeasuringPointLongText");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<MeasuringPointLongText> createMeasuringPointLongText(
        @Nonnull final MeasuringPointLongText measuringPointLongText )
    {
        return new CreateRequestBuilder<MeasuringPointLongText>(
            servicePath,
            measuringPointLongText,
            "MeasuringPointLongText");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<MeasuringPointLongText> updateMeasuringPointLongText(
        @Nonnull final MeasuringPointLongText measuringPointLongText )
    {
        return new UpdateRequestBuilder<MeasuringPointLongText>(
            servicePath,
            measuringPointLongText,
            "MeasuringPointLongText");
    }

}
