/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.DeleteRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.productexclusion.ProdSubstnExclsn;

import lombok.Getter;

/**
 * <p>
 * This service enables you to create product exclusions based on various factors like time range validities,
 * plant-specific, and application-specific. Additionally, predefined groups can be applied on these product exclusions.
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/98/013b603ed54532a5a6c3ea7cd975be/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/PRODUCTEXCLUSION_0001?section=OVERVIEW'>SAP API
 * Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>PRODUCTEXCLUSION_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Master Data - Product Exclusion Integration (SAP_COM_0708)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/1JW'>Advanced Available-to-Promise Processing (1JW)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_PRODSUBSTNEXCLSN</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>ProductExclusion</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultProductExclusionService" )
public class DefaultProductExclusionService implements ServiceWithNavigableEntities, ProductExclusionService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link ProductExclusionService#DEFAULT_SERVICE_PATH} to send the requests.
     * 
     */
    public DefaultProductExclusionService()
    {
        servicePath = ProductExclusionService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultProductExclusionService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultProductExclusionService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultProductExclusionService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<ProdSubstnExclsn> getAllProdSubstnExclsn()
    {
        return new GetAllRequestBuilder<ProdSubstnExclsn>(servicePath, ProdSubstnExclsn.class, "A_ProdSubstnExclsn");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<ProdSubstnExclsn> countProdSubstnExclsn()
    {
        return new CountRequestBuilder<ProdSubstnExclsn>(servicePath, ProdSubstnExclsn.class, "A_ProdSubstnExclsn");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<ProdSubstnExclsn> getProdSubstnExclsnByKey( @Nonnull final UUID mDSubstnUUID )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("MDSubstnUUID", mDSubstnUUID);
        return new GetByKeyRequestBuilder<ProdSubstnExclsn>(
            servicePath,
            ProdSubstnExclsn.class,
            key,
            "A_ProdSubstnExclsn");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<ProdSubstnExclsn> createProdSubstnExclsn(
        @Nonnull final ProdSubstnExclsn prodSubstnExclsn )
    {
        return new CreateRequestBuilder<ProdSubstnExclsn>(servicePath, prodSubstnExclsn, "A_ProdSubstnExclsn");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<ProdSubstnExclsn> updateProdSubstnExclsn(
        @Nonnull final ProdSubstnExclsn prodSubstnExclsn )
    {
        return new UpdateRequestBuilder<ProdSubstnExclsn>(servicePath, prodSubstnExclsn, "A_ProdSubstnExclsn");
    }

    @Override
    @Nonnull
    public DeleteRequestBuilder<ProdSubstnExclsn> deleteProdSubstnExclsn(
        @Nonnull final ProdSubstnExclsn prodSubstnExclsn )
    {
        return new DeleteRequestBuilder<ProdSubstnExclsn>(servicePath, prodSubstnExclsn, "A_ProdSubstnExclsn");
    }

}
