/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.productgroupdata.ProductGroup;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.productgroupdata.ProductGroupText;

import lombok.Getter;

/**
 * <p>
 * 
 * This synchronous inbound V4 OData service enables external systems and other consumers to read product group data and
 * their descriptions. This service only supports the GET operation. .
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/27/9eb92259584b1891753250bc3ae320/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference:
 * <a href='https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/PRODUCTGROUP_0001?section=OVERVIEW'>SAP
 * API Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>PRODUCTGROUP_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Product Integration (SAP_COM_0009)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/1RO'>Master Data Integration (1RO)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/1VP'>Opportunity-to-Order with SAP Cloud for Customer (1VP)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509, OAuth2</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_PRODUCTGROUP_2</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>ProductGroup</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultProductGroupDataService" )
public class DefaultProductGroupDataService implements ServiceWithNavigableEntities, ProductGroupDataService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link ProductGroupDataService#DEFAULT_SERVICE_PATH} to send the requests.
     * 
     */
    public DefaultProductGroupDataService()
    {
        servicePath = ProductGroupDataService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultProductGroupDataService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultProductGroupDataService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultProductGroupDataService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<ProductGroup> getAllProductGroup()
    {
        return new GetAllRequestBuilder<ProductGroup>(servicePath, ProductGroup.class, "ProductGroup");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<ProductGroup> countProductGroup()
    {
        return new CountRequestBuilder<ProductGroup>(servicePath, ProductGroup.class, "ProductGroup");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<ProductGroup> getProductGroupByKey( @Nonnull final String productGroup )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("ProductGroup", productGroup);
        return new GetByKeyRequestBuilder<ProductGroup>(servicePath, ProductGroup.class, key, "ProductGroup");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<ProductGroupText> getAllProductGroupText()
    {
        return new GetAllRequestBuilder<ProductGroupText>(servicePath, ProductGroupText.class, "ProductGroupText");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<ProductGroupText> countProductGroupText()
    {
        return new CountRequestBuilder<ProductGroupText>(servicePath, ProductGroupText.class, "ProductGroupText");
    }

    @Override
    @Nonnull
    public
        GetByKeyRequestBuilder<ProductGroupText>
        getProductGroupTextByKey( @Nonnull final String productGroup, @Nonnull final String language )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("ProductGroup", productGroup);
        key.put("Language", language);
        return new GetByKeyRequestBuilder<ProductGroupText>(
            servicePath,
            ProductGroupText.class,
            key,
            "ProductGroupText");
    }

}
