/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.time.LocalDate;
import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.producthierarchynode.ProdUnivHierNodeByHierID;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.producthierarchynode.ProdUniversalHierarchy;

import lombok.Getter;

/**
 * <p>
 * 
 * This synchronous inbound OData service enables external systems and other consumers to read currently active version
 * for a given Hierarchy ID, or the version with a specified validity date.
 * 
 * You can also use this service to read the details of a particular hierarchy, such as the node, its parent, its
 * children and other attributes of a node, and so on. This information is used in the processes related to sales and
 * distribution..
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/ce/d04ba703d248f88b862e22cb3b536b/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/PRODHIERNODES_0001?section=OVERVIEW'>SAP API
 * Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>PRODHIERNODES_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Product Integration (SAP_COM_0009)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/1RO'>Master Data Integration (1RO)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/1VP'>Opportunity-to-Order with SAP Cloud for Customer (1VP)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509, OAuth2</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_PROD_HIER_NODES_SRV</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>Product</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultProductHierarchyNodeService" )
public class DefaultProductHierarchyNodeService implements ServiceWithNavigableEntities, ProductHierarchyNodeService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link ProductHierarchyNodeService#DEFAULT_SERVICE_PATH} to send the requests.
     * 
     */
    public DefaultProductHierarchyNodeService()
    {
        servicePath = ProductHierarchyNodeService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultProductHierarchyNodeService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultProductHierarchyNodeService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultProductHierarchyNodeService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<ProdUniversalHierarchy> getAllProdUniversalHierarchy()
    {
        return new GetAllRequestBuilder<ProdUniversalHierarchy>(
            servicePath,
            ProdUniversalHierarchy.class,
            "ProdUniversalHierarchy");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<ProdUniversalHierarchy> countProdUniversalHierarchy()
    {
        return new CountRequestBuilder<ProdUniversalHierarchy>(
            servicePath,
            ProdUniversalHierarchy.class,
            "ProdUniversalHierarchy");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<ProdUniversalHierarchy> getProdUniversalHierarchyByKey(
        @Nonnull final String prodUnivHierarchy,
        @Nonnull final LocalDate prodHierarchyValidityEndDate )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("ProdUnivHierarchy", prodUnivHierarchy);
        key.put("ProdHierarchyValidityEndDate", prodHierarchyValidityEndDate);
        return new GetByKeyRequestBuilder<ProdUniversalHierarchy>(
            servicePath,
            ProdUniversalHierarchy.class,
            key,
            "ProdUniversalHierarchy");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<ProdUnivHierNodeByHierID> getAllProdUnivHierNodeByHierID()
    {
        return new GetAllRequestBuilder<ProdUnivHierNodeByHierID>(
            servicePath,
            ProdUnivHierNodeByHierID.class,
            "ProdUnivHierNodeByHierID");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<ProdUnivHierNodeByHierID> countProdUnivHierNodeByHierID()
    {
        return new CountRequestBuilder<ProdUnivHierNodeByHierID>(
            servicePath,
            ProdUnivHierNodeByHierID.class,
            "ProdUnivHierNodeByHierID");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<ProdUnivHierNodeByHierID> getProdUnivHierNodeByHierIDByKey(
        @Nonnull final String prodUnivHierarchy,
        @Nonnull final String hierarchyNode,
        @Nonnull final LocalDate prodHierarchyValidityEndDate )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("ProdUnivHierarchy", prodUnivHierarchy);
        key.put("HierarchyNode", hierarchyNode);
        key.put("ProdHierarchyValidityEndDate", prodHierarchyValidityEndDate);
        return new GetByKeyRequestBuilder<ProdUnivHierNodeByHierID>(
            servicePath,
            ProdUnivHierNodeByHierID.class,
            key,
            "ProdUnivHierNodeByHierID");
    }

}
