/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.DeleteRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.productsubstitution.ProdSubstn;

import lombok.Getter;

/**
 * <p>
 * This service enables you to create plant, application and time range specific product substitution. Additionally,
 * these substitutions can be sequenced, and grouped using predefined groups.
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/4e/7fa2c048e041cd8430a7484c4f38f0/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/PRODUCTSUBSTITUTION_0001?section=OVERVIEW'>SAP
 * API Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>PRODUCTSUBSTITUTION_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Master Data - Product Substitution Integration (SAP_COM_0707)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/1JW'>Advanced Available-to-Promise Processing (1JW)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_PRODUCTSUBSTITUTION</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>ProductSubstitution</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultProductSubstitutionService" )
public class DefaultProductSubstitutionService implements ServiceWithNavigableEntities, ProductSubstitutionService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link ProductSubstitutionService#DEFAULT_SERVICE_PATH} to send the requests.
     * 
     */
    public DefaultProductSubstitutionService()
    {
        servicePath = ProductSubstitutionService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultProductSubstitutionService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultProductSubstitutionService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultProductSubstitutionService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<ProdSubstn> getAllProdSubstn()
    {
        return new GetAllRequestBuilder<ProdSubstn>(servicePath, ProdSubstn.class, "A_ProdSubstn");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<ProdSubstn> countProdSubstn()
    {
        return new CountRequestBuilder<ProdSubstn>(servicePath, ProdSubstn.class, "A_ProdSubstn");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<ProdSubstn> getProdSubstnByKey( @Nonnull final UUID mDSubstnUUID )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("MDSubstnUUID", mDSubstnUUID);
        return new GetByKeyRequestBuilder<ProdSubstn>(servicePath, ProdSubstn.class, key, "A_ProdSubstn");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<ProdSubstn> createProdSubstn( @Nonnull final ProdSubstn prodSubstn )
    {
        return new CreateRequestBuilder<ProdSubstn>(servicePath, prodSubstn, "A_ProdSubstn");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<ProdSubstn> updateProdSubstn( @Nonnull final ProdSubstn prodSubstn )
    {
        return new UpdateRequestBuilder<ProdSubstn>(servicePath, prodSubstn, "A_ProdSubstn");
    }

    @Override
    @Nonnull
    public DeleteRequestBuilder<ProdSubstn> deleteProdSubstn( @Nonnull final ProdSubstn prodSubstn )
    {
        return new DeleteRequestBuilder<ProdSubstn>(servicePath, prodSubstn, "A_ProdSubstn");
    }

}
