/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.time.LocalDate;
import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.productstoproducthierarchiesassignment.ProdUnivHierNormalNode;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.productstoproducthierarchiesassignment.ProdUnivHierProdByHierNode;

import lombok.Getter;

/**
 * <p>
 * 
 * This service enables you to read the products assigned to product hierarchy details by exposing the remote API views
 * through OData. These products are assigned to the corresponding product hierarchy based on the validity of the
 * hierarchy version. This is a synchronous service. .
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/cf/e507152a7b47f3bd2d8626c365468d/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/PRODHIERPRODS_0001?section=OVERVIEW'>SAP API
 * Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>PRODHIERPRODS_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Product Integration (SAP_COM_0009)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/1RO'>Master Data Integration (1RO)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/1VP'>Opportunity-to-Order with SAP Cloud for Customer (1VP)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509, OAuth2</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_PROD_HIER_PRODS_SRV</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>Product</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultProductsToProductHierarchiesAssignmentService" )
public class DefaultProductsToProductHierarchiesAssignmentService
    implements
    ServiceWithNavigableEntities,
    ProductsToProductHierarchiesAssignmentService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link ProductsToProductHierarchiesAssignmentService#DEFAULT_SERVICE_PATH} to send the
     * requests.
     * 
     */
    public DefaultProductsToProductHierarchiesAssignmentService()
    {
        servicePath = ProductsToProductHierarchiesAssignmentService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultProductsToProductHierarchiesAssignmentService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultProductsToProductHierarchiesAssignmentService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultProductsToProductHierarchiesAssignmentService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<ProdUnivHierNormalNode> getAllProdUnivHierNormalNode()
    {
        return new GetAllRequestBuilder<ProdUnivHierNormalNode>(
            servicePath,
            ProdUnivHierNormalNode.class,
            "ProdUnivHierNormalNode");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<ProdUnivHierNormalNode> countProdUnivHierNormalNode()
    {
        return new CountRequestBuilder<ProdUnivHierNormalNode>(
            servicePath,
            ProdUnivHierNormalNode.class,
            "ProdUnivHierNormalNode");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<ProdUnivHierNormalNode> getProdUnivHierNormalNodeByKey(
        @Nonnull final String prodUnivHierarchy,
        @Nonnull final String hierarchyNode,
        @Nonnull final LocalDate prodHierarchyValidityEndDate )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("ProdUnivHierarchy", prodUnivHierarchy);
        key.put("HierarchyNode", hierarchyNode);
        key.put("ProdHierarchyValidityEndDate", prodHierarchyValidityEndDate);
        return new GetByKeyRequestBuilder<ProdUnivHierNormalNode>(
            servicePath,
            ProdUnivHierNormalNode.class,
            key,
            "ProdUnivHierNormalNode");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<ProdUnivHierProdByHierNode> getAllProdUnivHierProdByHierNode()
    {
        return new GetAllRequestBuilder<ProdUnivHierProdByHierNode>(
            servicePath,
            ProdUnivHierProdByHierNode.class,
            "ProdUnivHierProdByHierNode");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<ProdUnivHierProdByHierNode> countProdUnivHierProdByHierNode()
    {
        return new CountRequestBuilder<ProdUnivHierProdByHierNode>(
            servicePath,
            ProdUnivHierProdByHierNode.class,
            "ProdUnivHierProdByHierNode");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<ProdUnivHierProdByHierNode> getProdUnivHierProdByHierNodeByKey(
        @Nonnull final String prodUnivHierarchy,
        @Nonnull final String hierarchyNode,
        @Nonnull final LocalDate prodHierarchyValidityEndDate,
        @Nonnull final String product )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("ProdUnivHierarchy", prodUnivHierarchy);
        key.put("HierarchyNode", hierarchyNode);
        key.put("ProdHierarchyValidityEndDate", prodHierarchyValidityEndDate);
        key.put("Product", product);
        return new GetByKeyRequestBuilder<ProdUnivHierProdByHierNode>(
            servicePath,
            ProdUnivHierProdByHierNode.class,
            key,
            "ProdUnivHierProdByHierNode");
    }

}
