/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.time.LocalDate;
import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.DeleteRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.realestatecontract.REContrAdjustmentTerm;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.realestatecontract.REContrCondition;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.realestatecontract.REContrNoticeTerm;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.realestatecontract.REContrNoticeTermForObj;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.realestatecontract.REContrObjAssgmt;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.realestatecontract.REContrOrglAssgmtTerm;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.realestatecontract.REContrPartAssgmt;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.realestatecontract.REContrPostingTerm;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.realestatecontract.REContrReminderDate;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.realestatecontract.REContrReminderRule;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.realestatecontract.REContrRenewalTerm;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.realestatecontract.REContrRhythmTerm;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.realestatecontract.REContrValuation;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.realestatecontract.REContrValuationCondition;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.realestatecontract.REContract;

import lombok.Getter;

/**
 * <p>
 * 
 * The service contains contract header and sub nodes for managing the contract process. These child objects are
 * Business Partner Assignments, Object Assignments, Posting, Rhythm, Adjustment, Renewal, Notice and Organizational
 * Assignment Terms, Conditions, Reminder Rules and Dates, Valuation Rules and Condition Valuation Assignments. The
 * contract header can be read, created and updated, additionally all child objects can be deleted using the service.
 * The contract header numbers (InternalRealEstateNumber and RealEstateContract) must be fetched via read request after
 * creating due to late numbering mechanism. If there are any issues when the contract header or any object is created,
 * updated or deleted, the system displays error messages in the response. .
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/1f/9493a0f6824ca89ca3fd458e0ceaa2/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/API_RECONTRACT_0001?section=OVERVIEW'>SAP API
 * Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>API_RECONTRACT_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>SAP Real Estate Contract API Integration (SAP_COM_0393)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/1T6'>Lease-In Accounting (1T6)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/21Q'>Lease-Out Accounting (21Q)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/21R'>Service Contract (21R)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_REAL_ESTATE_CONTRACT</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>RealEstateContract</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultRealEstateContractService" )
public class DefaultRealEstateContractService implements ServiceWithNavigableEntities, RealEstateContractService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link RealEstateContractService#DEFAULT_SERVICE_PATH} to send the requests.
     * 
     */
    public DefaultRealEstateContractService()
    {
        servicePath = RealEstateContractService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultRealEstateContractService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultRealEstateContractService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultRealEstateContractService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<REContract> getAllREContract()
    {
        return new GetAllRequestBuilder<REContract>(servicePath, REContract.class, "A_REContract");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<REContract> countREContract()
    {
        return new CountRequestBuilder<REContract>(servicePath, REContract.class, "A_REContract");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<REContract> getREContractByKey( @Nonnull final String internalRealEstateNumber )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("InternalRealEstateNumber", internalRealEstateNumber);
        return new GetByKeyRequestBuilder<REContract>(servicePath, REContract.class, key, "A_REContract");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<REContract> createREContract( @Nonnull final REContract rEContract )
    {
        return new CreateRequestBuilder<REContract>(servicePath, rEContract, "A_REContract");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<REContract> updateREContract( @Nonnull final REContract rEContract )
    {
        return new UpdateRequestBuilder<REContract>(servicePath, rEContract, "A_REContract");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<REContrAdjustmentTerm> getAllREContrAdjustmentTerm()
    {
        return new GetAllRequestBuilder<REContrAdjustmentTerm>(
            servicePath,
            REContrAdjustmentTerm.class,
            "A_REContrAdjustmentTerm");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<REContrAdjustmentTerm> countREContrAdjustmentTerm()
    {
        return new CountRequestBuilder<REContrAdjustmentTerm>(
            servicePath,
            REContrAdjustmentTerm.class,
            "A_REContrAdjustmentTerm");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<REContrAdjustmentTerm> getREContrAdjustmentTermByKey(
        @Nonnull final String internalRealEstateNumber,
        @Nonnull final String rETermNumber,
        @Nonnull final String rEAdjustmentRule,
        @Nonnull final String rEAdjustmentElementaryRule,
        @Nonnull final String validityStartEndDateValue )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("InternalRealEstateNumber", internalRealEstateNumber);
        key.put("RETermNumber", rETermNumber);
        key.put("REAdjustmentRule", rEAdjustmentRule);
        key.put("REAdjustmentElementaryRule", rEAdjustmentElementaryRule);
        key.put("ValidityStartEndDateValue", validityStartEndDateValue);
        return new GetByKeyRequestBuilder<REContrAdjustmentTerm>(
            servicePath,
            REContrAdjustmentTerm.class,
            key,
            "A_REContrAdjustmentTerm");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<REContrAdjustmentTerm> createREContrAdjustmentTerm(
        @Nonnull final REContrAdjustmentTerm rEContrAdjustmentTerm )
    {
        return new CreateRequestBuilder<REContrAdjustmentTerm>(
            servicePath,
            rEContrAdjustmentTerm,
            "A_REContrAdjustmentTerm");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<REContrAdjustmentTerm> updateREContrAdjustmentTerm(
        @Nonnull final REContrAdjustmentTerm rEContrAdjustmentTerm )
    {
        return new UpdateRequestBuilder<REContrAdjustmentTerm>(
            servicePath,
            rEContrAdjustmentTerm,
            "A_REContrAdjustmentTerm");
    }

    @Override
    @Nonnull
    public DeleteRequestBuilder<REContrAdjustmentTerm> deleteREContrAdjustmentTerm(
        @Nonnull final REContrAdjustmentTerm rEContrAdjustmentTerm )
    {
        return new DeleteRequestBuilder<REContrAdjustmentTerm>(
            servicePath,
            rEContrAdjustmentTerm,
            "A_REContrAdjustmentTerm");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<REContrCondition> getAllREContrCondition()
    {
        return new GetAllRequestBuilder<REContrCondition>(servicePath, REContrCondition.class, "A_REContrCondition");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<REContrCondition> countREContrCondition()
    {
        return new CountRequestBuilder<REContrCondition>(servicePath, REContrCondition.class, "A_REContrCondition");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<REContrCondition> getREContrConditionByKey(
        @Nonnull final String internalRealEstateNumber,
        @Nonnull final String rEStatusObjectCalculation,
        @Nonnull final String rEConditionType,
        @Nonnull final String rEExtConditionPurpose,
        @Nonnull final String validityStartEndDateValue )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("InternalRealEstateNumber", internalRealEstateNumber);
        key.put("REStatusObjectCalculation", rEStatusObjectCalculation);
        key.put("REConditionType", rEConditionType);
        key.put("REExtConditionPurpose", rEExtConditionPurpose);
        key.put("ValidityStartEndDateValue", validityStartEndDateValue);
        return new GetByKeyRequestBuilder<REContrCondition>(
            servicePath,
            REContrCondition.class,
            key,
            "A_REContrCondition");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<REContrCondition> createREContrCondition(
        @Nonnull final REContrCondition rEContrCondition )
    {
        return new CreateRequestBuilder<REContrCondition>(servicePath, rEContrCondition, "A_REContrCondition");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<REContrCondition> updateREContrCondition(
        @Nonnull final REContrCondition rEContrCondition )
    {
        return new UpdateRequestBuilder<REContrCondition>(servicePath, rEContrCondition, "A_REContrCondition");
    }

    @Override
    @Nonnull
    public DeleteRequestBuilder<REContrCondition> deleteREContrCondition(
        @Nonnull final REContrCondition rEContrCondition )
    {
        return new DeleteRequestBuilder<REContrCondition>(servicePath, rEContrCondition, "A_REContrCondition");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<REContrNoticeTerm> getAllREContrNoticeTerm()
    {
        return new GetAllRequestBuilder<REContrNoticeTerm>(servicePath, REContrNoticeTerm.class, "A_REContrNoticeTerm");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<REContrNoticeTerm> countREContrNoticeTerm()
    {
        return new CountRequestBuilder<REContrNoticeTerm>(servicePath, REContrNoticeTerm.class, "A_REContrNoticeTerm");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<REContrNoticeTerm> getREContrNoticeTermByKey(
        @Nonnull final String internalRealEstateNumber,
        @Nonnull final String rETermType,
        @Nonnull final String rETermNumber,
        @Nonnull final String rENoticeRule,
        @Nonnull final String rENoticeSequenceNo )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("InternalRealEstateNumber", internalRealEstateNumber);
        key.put("RETermType", rETermType);
        key.put("RETermNumber", rETermNumber);
        key.put("RENoticeRule", rENoticeRule);
        key.put("RENoticeSequenceNo", rENoticeSequenceNo);
        return new GetByKeyRequestBuilder<REContrNoticeTerm>(
            servicePath,
            REContrNoticeTerm.class,
            key,
            "A_REContrNoticeTerm");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<REContrNoticeTerm> createREContrNoticeTerm(
        @Nonnull final REContrNoticeTerm rEContrNoticeTerm )
    {
        return new CreateRequestBuilder<REContrNoticeTerm>(servicePath, rEContrNoticeTerm, "A_REContrNoticeTerm");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<REContrNoticeTerm> updateREContrNoticeTerm(
        @Nonnull final REContrNoticeTerm rEContrNoticeTerm )
    {
        return new UpdateRequestBuilder<REContrNoticeTerm>(servicePath, rEContrNoticeTerm, "A_REContrNoticeTerm");
    }

    @Override
    @Nonnull
    public DeleteRequestBuilder<REContrNoticeTerm> deleteREContrNoticeTerm(
        @Nonnull final REContrNoticeTerm rEContrNoticeTerm )
    {
        return new DeleteRequestBuilder<REContrNoticeTerm>(servicePath, rEContrNoticeTerm, "A_REContrNoticeTerm");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<REContrNoticeTermForObj> getAllREContrNoticeTermForObj()
    {
        return new GetAllRequestBuilder<REContrNoticeTermForObj>(
            servicePath,
            REContrNoticeTermForObj.class,
            "A_REContrNoticeTermForObj");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<REContrNoticeTermForObj> countREContrNoticeTermForObj()
    {
        return new CountRequestBuilder<REContrNoticeTermForObj>(
            servicePath,
            REContrNoticeTermForObj.class,
            "A_REContrNoticeTermForObj");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<REContrNoticeTermForObj> getREContrNoticeTermForObjByKey(
        @Nonnull final String internalRealEstateNumber,
        @Nonnull final String rETermType,
        @Nonnull final String rEStatusObject,
        @Nonnull final String rETermNumber,
        @Nonnull final String rENoticeRule,
        @Nonnull final String rENoticeSequenceNo )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("InternalRealEstateNumber", internalRealEstateNumber);
        key.put("RETermType", rETermType);
        key.put("REStatusObject", rEStatusObject);
        key.put("RETermNumber", rETermNumber);
        key.put("RENoticeRule", rENoticeRule);
        key.put("RENoticeSequenceNo", rENoticeSequenceNo);
        return new GetByKeyRequestBuilder<REContrNoticeTermForObj>(
            servicePath,
            REContrNoticeTermForObj.class,
            key,
            "A_REContrNoticeTermForObj");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<REContrNoticeTermForObj> createREContrNoticeTermForObj(
        @Nonnull final REContrNoticeTermForObj rEContrNoticeTermForObj )
    {
        return new CreateRequestBuilder<REContrNoticeTermForObj>(
            servicePath,
            rEContrNoticeTermForObj,
            "A_REContrNoticeTermForObj");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<REContrNoticeTermForObj> updateREContrNoticeTermForObj(
        @Nonnull final REContrNoticeTermForObj rEContrNoticeTermForObj )
    {
        return new UpdateRequestBuilder<REContrNoticeTermForObj>(
            servicePath,
            rEContrNoticeTermForObj,
            "A_REContrNoticeTermForObj");
    }

    @Override
    @Nonnull
    public DeleteRequestBuilder<REContrNoticeTermForObj> deleteREContrNoticeTermForObj(
        @Nonnull final REContrNoticeTermForObj rEContrNoticeTermForObj )
    {
        return new DeleteRequestBuilder<REContrNoticeTermForObj>(
            servicePath,
            rEContrNoticeTermForObj,
            "A_REContrNoticeTermForObj");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<REContrObjAssgmt> getAllREContrObjAssgmt()
    {
        return new GetAllRequestBuilder<REContrObjAssgmt>(servicePath, REContrObjAssgmt.class, "A_REContrObjAssgmt");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<REContrObjAssgmt> countREContrObjAssgmt()
    {
        return new CountRequestBuilder<REContrObjAssgmt>(servicePath, REContrObjAssgmt.class, "A_REContrObjAssgmt");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<REContrObjAssgmt> getREContrObjAssgmtByKey(
        @Nonnull final String rEStatusObjectSource,
        @Nonnull final String rEObjectAssignmentType,
        @Nonnull final String rEStatusObjectTarget,
        @Nonnull final String validityStartEndDateValue,
        @Nonnull final String internalRealEstateNumber )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("REStatusObjectSource", rEStatusObjectSource);
        key.put("REObjectAssignmentType", rEObjectAssignmentType);
        key.put("REStatusObjectTarget", rEStatusObjectTarget);
        key.put("ValidityStartEndDateValue", validityStartEndDateValue);
        key.put("InternalRealEstateNumber", internalRealEstateNumber);
        return new GetByKeyRequestBuilder<REContrObjAssgmt>(
            servicePath,
            REContrObjAssgmt.class,
            key,
            "A_REContrObjAssgmt");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<REContrObjAssgmt> createREContrObjAssgmt(
        @Nonnull final REContrObjAssgmt rEContrObjAssgmt )
    {
        return new CreateRequestBuilder<REContrObjAssgmt>(servicePath, rEContrObjAssgmt, "A_REContrObjAssgmt");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<REContrObjAssgmt> updateREContrObjAssgmt(
        @Nonnull final REContrObjAssgmt rEContrObjAssgmt )
    {
        return new UpdateRequestBuilder<REContrObjAssgmt>(servicePath, rEContrObjAssgmt, "A_REContrObjAssgmt");
    }

    @Override
    @Nonnull
    public DeleteRequestBuilder<REContrObjAssgmt> deleteREContrObjAssgmt(
        @Nonnull final REContrObjAssgmt rEContrObjAssgmt )
    {
        return new DeleteRequestBuilder<REContrObjAssgmt>(servicePath, rEContrObjAssgmt, "A_REContrObjAssgmt");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<REContrOrglAssgmtTerm> getAllREContrOrglAssgmtTerm()
    {
        return new GetAllRequestBuilder<REContrOrglAssgmtTerm>(
            servicePath,
            REContrOrglAssgmtTerm.class,
            "A_REContrOrglAssgmtTerm");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<REContrOrglAssgmtTerm> countREContrOrglAssgmtTerm()
    {
        return new CountRequestBuilder<REContrOrglAssgmtTerm>(
            servicePath,
            REContrOrglAssgmtTerm.class,
            "A_REContrOrglAssgmtTerm");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<REContrOrglAssgmtTerm> getREContrOrglAssgmtTermByKey(
        @Nonnull final String internalRealEstateNumber,
        @Nonnull final String rETermType,
        @Nonnull final String rETermNumber,
        @Nonnull final String validityStartEndDateValue )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("InternalRealEstateNumber", internalRealEstateNumber);
        key.put("RETermType", rETermType);
        key.put("RETermNumber", rETermNumber);
        key.put("ValidityStartEndDateValue", validityStartEndDateValue);
        return new GetByKeyRequestBuilder<REContrOrglAssgmtTerm>(
            servicePath,
            REContrOrglAssgmtTerm.class,
            key,
            "A_REContrOrglAssgmtTerm");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<REContrOrglAssgmtTerm> createREContrOrglAssgmtTerm(
        @Nonnull final REContrOrglAssgmtTerm rEContrOrglAssgmtTerm )
    {
        return new CreateRequestBuilder<REContrOrglAssgmtTerm>(
            servicePath,
            rEContrOrglAssgmtTerm,
            "A_REContrOrglAssgmtTerm");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<REContrOrglAssgmtTerm> updateREContrOrglAssgmtTerm(
        @Nonnull final REContrOrglAssgmtTerm rEContrOrglAssgmtTerm )
    {
        return new UpdateRequestBuilder<REContrOrglAssgmtTerm>(
            servicePath,
            rEContrOrglAssgmtTerm,
            "A_REContrOrglAssgmtTerm");
    }

    @Override
    @Nonnull
    public DeleteRequestBuilder<REContrOrglAssgmtTerm> deleteREContrOrglAssgmtTerm(
        @Nonnull final REContrOrglAssgmtTerm rEContrOrglAssgmtTerm )
    {
        return new DeleteRequestBuilder<REContrOrglAssgmtTerm>(
            servicePath,
            rEContrOrglAssgmtTerm,
            "A_REContrOrglAssgmtTerm");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<REContrPartAssgmt> getAllREContrPartAssgmt()
    {
        return new GetAllRequestBuilder<REContrPartAssgmt>(servicePath, REContrPartAssgmt.class, "A_REContrPartAssgmt");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<REContrPartAssgmt> countREContrPartAssgmt()
    {
        return new CountRequestBuilder<REContrPartAssgmt>(servicePath, REContrPartAssgmt.class, "A_REContrPartAssgmt");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<REContrPartAssgmt> getREContrPartAssgmtByKey(
        @Nonnull final String internalRealEstateNumber,
        @Nonnull final String businessPartner,
        @Nonnull final String businessPartnerRole,
        @Nonnull final String validityStartEndDateValue )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("InternalRealEstateNumber", internalRealEstateNumber);
        key.put("BusinessPartner", businessPartner);
        key.put("BusinessPartnerRole", businessPartnerRole);
        key.put("ValidityStartEndDateValue", validityStartEndDateValue);
        return new GetByKeyRequestBuilder<REContrPartAssgmt>(
            servicePath,
            REContrPartAssgmt.class,
            key,
            "A_REContrPartAssgmt");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<REContrPartAssgmt> createREContrPartAssgmt(
        @Nonnull final REContrPartAssgmt rEContrPartAssgmt )
    {
        return new CreateRequestBuilder<REContrPartAssgmt>(servicePath, rEContrPartAssgmt, "A_REContrPartAssgmt");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<REContrPartAssgmt> updateREContrPartAssgmt(
        @Nonnull final REContrPartAssgmt rEContrPartAssgmt )
    {
        return new UpdateRequestBuilder<REContrPartAssgmt>(servicePath, rEContrPartAssgmt, "A_REContrPartAssgmt");
    }

    @Override
    @Nonnull
    public DeleteRequestBuilder<REContrPartAssgmt> deleteREContrPartAssgmt(
        @Nonnull final REContrPartAssgmt rEContrPartAssgmt )
    {
        return new DeleteRequestBuilder<REContrPartAssgmt>(servicePath, rEContrPartAssgmt, "A_REContrPartAssgmt");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<REContrPostingTerm> getAllREContrPostingTerm()
    {
        return new GetAllRequestBuilder<REContrPostingTerm>(
            servicePath,
            REContrPostingTerm.class,
            "A_REContrPostingTerm");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<REContrPostingTerm> countREContrPostingTerm()
    {
        return new CountRequestBuilder<REContrPostingTerm>(
            servicePath,
            REContrPostingTerm.class,
            "A_REContrPostingTerm");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<REContrPostingTerm> getREContrPostingTermByKey(
        @Nonnull final String internalRealEstateNumber,
        @Nonnull final String rETermType,
        @Nonnull final String rETermNumber,
        @Nonnull final String validityStartEndDateValue )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("InternalRealEstateNumber", internalRealEstateNumber);
        key.put("RETermType", rETermType);
        key.put("RETermNumber", rETermNumber);
        key.put("ValidityStartEndDateValue", validityStartEndDateValue);
        return new GetByKeyRequestBuilder<REContrPostingTerm>(
            servicePath,
            REContrPostingTerm.class,
            key,
            "A_REContrPostingTerm");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<REContrPostingTerm> createREContrPostingTerm(
        @Nonnull final REContrPostingTerm rEContrPostingTerm )
    {
        return new CreateRequestBuilder<REContrPostingTerm>(servicePath, rEContrPostingTerm, "A_REContrPostingTerm");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<REContrPostingTerm> updateREContrPostingTerm(
        @Nonnull final REContrPostingTerm rEContrPostingTerm )
    {
        return new UpdateRequestBuilder<REContrPostingTerm>(servicePath, rEContrPostingTerm, "A_REContrPostingTerm");
    }

    @Override
    @Nonnull
    public DeleteRequestBuilder<REContrPostingTerm> deleteREContrPostingTerm(
        @Nonnull final REContrPostingTerm rEContrPostingTerm )
    {
        return new DeleteRequestBuilder<REContrPostingTerm>(servicePath, rEContrPostingTerm, "A_REContrPostingTerm");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<REContrReminderDate> getAllREContrReminderDate()
    {
        return new GetAllRequestBuilder<REContrReminderDate>(
            servicePath,
            REContrReminderDate.class,
            "A_REContrReminderDate");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<REContrReminderDate> countREContrReminderDate()
    {
        return new CountRequestBuilder<REContrReminderDate>(
            servicePath,
            REContrReminderDate.class,
            "A_REContrReminderDate");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<REContrReminderDate> getREContrReminderDateByKey(
        @Nonnull final String internalRealEstateNumber,
        @Nonnull final String rEReminderNumber,
        @Nonnull final LocalDate rEReminderDate )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("InternalRealEstateNumber", internalRealEstateNumber);
        key.put("REReminderNumber", rEReminderNumber);
        key.put("REReminderDate", rEReminderDate);
        return new GetByKeyRequestBuilder<REContrReminderDate>(
            servicePath,
            REContrReminderDate.class,
            key,
            "A_REContrReminderDate");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<REContrReminderDate> updateREContrReminderDate(
        @Nonnull final REContrReminderDate rEContrReminderDate )
    {
        return new UpdateRequestBuilder<REContrReminderDate>(servicePath, rEContrReminderDate, "A_REContrReminderDate");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<REContrReminderRule> getAllREContrReminderRule()
    {
        return new GetAllRequestBuilder<REContrReminderRule>(
            servicePath,
            REContrReminderRule.class,
            "A_REContrReminderRule");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<REContrReminderRule> countREContrReminderRule()
    {
        return new CountRequestBuilder<REContrReminderRule>(
            servicePath,
            REContrReminderRule.class,
            "A_REContrReminderRule");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<REContrReminderRule> getREContrReminderRuleByKey(
        @Nonnull final String internalRealEstateNumber,
        @Nonnull final String rEReminderNumber,
        @Nonnull final String rEReminderRuleParamNumber )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("InternalRealEstateNumber", internalRealEstateNumber);
        key.put("REReminderNumber", rEReminderNumber);
        key.put("REReminderRuleParamNumber", rEReminderRuleParamNumber);
        return new GetByKeyRequestBuilder<REContrReminderRule>(
            servicePath,
            REContrReminderRule.class,
            key,
            "A_REContrReminderRule");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<REContrReminderRule> createREContrReminderRule(
        @Nonnull final REContrReminderRule rEContrReminderRule )
    {
        return new CreateRequestBuilder<REContrReminderRule>(servicePath, rEContrReminderRule, "A_REContrReminderRule");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<REContrReminderRule> updateREContrReminderRule(
        @Nonnull final REContrReminderRule rEContrReminderRule )
    {
        return new UpdateRequestBuilder<REContrReminderRule>(servicePath, rEContrReminderRule, "A_REContrReminderRule");
    }

    @Override
    @Nonnull
    public DeleteRequestBuilder<REContrReminderRule> deleteREContrReminderRule(
        @Nonnull final REContrReminderRule rEContrReminderRule )
    {
        return new DeleteRequestBuilder<REContrReminderRule>(servicePath, rEContrReminderRule, "A_REContrReminderRule");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<REContrRenewalTerm> getAllREContrRenewalTerm()
    {
        return new GetAllRequestBuilder<REContrRenewalTerm>(
            servicePath,
            REContrRenewalTerm.class,
            "A_REContrRenewalTerm");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<REContrRenewalTerm> countREContrRenewalTerm()
    {
        return new CountRequestBuilder<REContrRenewalTerm>(
            servicePath,
            REContrRenewalTerm.class,
            "A_REContrRenewalTerm");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<REContrRenewalTerm> getREContrRenewalTermByKey(
        @Nonnull final String internalRealEstateNumber,
        @Nonnull final String rETermType,
        @Nonnull final String rETermNumber,
        @Nonnull final String rERenewalType,
        @Nonnull final String rERenewalSequenceNumber,
        @Nonnull final String rERenewalRuleType )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("InternalRealEstateNumber", internalRealEstateNumber);
        key.put("RETermType", rETermType);
        key.put("RETermNumber", rETermNumber);
        key.put("RERenewalType", rERenewalType);
        key.put("RERenewalSequenceNumber", rERenewalSequenceNumber);
        key.put("RERenewalRuleType", rERenewalRuleType);
        return new GetByKeyRequestBuilder<REContrRenewalTerm>(
            servicePath,
            REContrRenewalTerm.class,
            key,
            "A_REContrRenewalTerm");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<REContrRenewalTerm> createREContrRenewalTerm(
        @Nonnull final REContrRenewalTerm rEContrRenewalTerm )
    {
        return new CreateRequestBuilder<REContrRenewalTerm>(servicePath, rEContrRenewalTerm, "A_REContrRenewalTerm");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<REContrRenewalTerm> updateREContrRenewalTerm(
        @Nonnull final REContrRenewalTerm rEContrRenewalTerm )
    {
        return new UpdateRequestBuilder<REContrRenewalTerm>(servicePath, rEContrRenewalTerm, "A_REContrRenewalTerm");
    }

    @Override
    @Nonnull
    public DeleteRequestBuilder<REContrRenewalTerm> deleteREContrRenewalTerm(
        @Nonnull final REContrRenewalTerm rEContrRenewalTerm )
    {
        return new DeleteRequestBuilder<REContrRenewalTerm>(servicePath, rEContrRenewalTerm, "A_REContrRenewalTerm");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<REContrRhythmTerm> getAllREContrRhythmTerm()
    {
        return new GetAllRequestBuilder<REContrRhythmTerm>(servicePath, REContrRhythmTerm.class, "A_REContrRhythmTerm");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<REContrRhythmTerm> countREContrRhythmTerm()
    {
        return new CountRequestBuilder<REContrRhythmTerm>(servicePath, REContrRhythmTerm.class, "A_REContrRhythmTerm");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<REContrRhythmTerm> getREContrRhythmTermByKey(
        @Nonnull final String internalRealEstateNumber,
        @Nonnull final String rETermType,
        @Nonnull final String rETermNumber,
        @Nonnull final String validityStartEndDateValue )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("InternalRealEstateNumber", internalRealEstateNumber);
        key.put("RETermType", rETermType);
        key.put("RETermNumber", rETermNumber);
        key.put("ValidityStartEndDateValue", validityStartEndDateValue);
        return new GetByKeyRequestBuilder<REContrRhythmTerm>(
            servicePath,
            REContrRhythmTerm.class,
            key,
            "A_REContrRhythmTerm");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<REContrRhythmTerm> createREContrRhythmTerm(
        @Nonnull final REContrRhythmTerm rEContrRhythmTerm )
    {
        return new CreateRequestBuilder<REContrRhythmTerm>(servicePath, rEContrRhythmTerm, "A_REContrRhythmTerm");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<REContrRhythmTerm> updateREContrRhythmTerm(
        @Nonnull final REContrRhythmTerm rEContrRhythmTerm )
    {
        return new UpdateRequestBuilder<REContrRhythmTerm>(servicePath, rEContrRhythmTerm, "A_REContrRhythmTerm");
    }

    @Override
    @Nonnull
    public DeleteRequestBuilder<REContrRhythmTerm> deleteREContrRhythmTerm(
        @Nonnull final REContrRhythmTerm rEContrRhythmTerm )
    {
        return new DeleteRequestBuilder<REContrRhythmTerm>(servicePath, rEContrRhythmTerm, "A_REContrRhythmTerm");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<REContrValuation> getAllREContrValuation()
    {
        return new GetAllRequestBuilder<REContrValuation>(servicePath, REContrValuation.class, "A_REContrValuation");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<REContrValuation> countREContrValuation()
    {
        return new CountRequestBuilder<REContrValuation>(servicePath, REContrValuation.class, "A_REContrValuation");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<REContrValuation> getREContrValuationByKey(
        @Nonnull final String internalRealEstateNumber,
        @Nonnull final String rETermNumber,
        @Nonnull final String validityStartEndDateValue )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("InternalRealEstateNumber", internalRealEstateNumber);
        key.put("RETermNumber", rETermNumber);
        key.put("ValidityStartEndDateValue", validityStartEndDateValue);
        return new GetByKeyRequestBuilder<REContrValuation>(
            servicePath,
            REContrValuation.class,
            key,
            "A_REContrValuation");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<REContrValuation> createREContrValuation(
        @Nonnull final REContrValuation rEContrValuation )
    {
        return new CreateRequestBuilder<REContrValuation>(servicePath, rEContrValuation, "A_REContrValuation");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<REContrValuation> updateREContrValuation(
        @Nonnull final REContrValuation rEContrValuation )
    {
        return new UpdateRequestBuilder<REContrValuation>(servicePath, rEContrValuation, "A_REContrValuation");
    }

    @Override
    @Nonnull
    public DeleteRequestBuilder<REContrValuation> deleteREContrValuation(
        @Nonnull final REContrValuation rEContrValuation )
    {
        return new DeleteRequestBuilder<REContrValuation>(servicePath, rEContrValuation, "A_REContrValuation");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<REContrValuationCondition> getAllREContrValuationCondition()
    {
        return new GetAllRequestBuilder<REContrValuationCondition>(
            servicePath,
            REContrValuationCondition.class,
            "A_REContrValuationCondition");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<REContrValuationCondition> countREContrValuationCondition()
    {
        return new CountRequestBuilder<REContrValuationCondition>(
            servicePath,
            REContrValuationCondition.class,
            "A_REContrValuationCondition");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<REContrValuationCondition> getREContrValuationConditionByKey(
        @Nonnull final String internalRealEstateNumber,
        @Nonnull final String rETermNumber,
        @Nonnull final String validityStartEndDateValue,
        @Nonnull final String rEConditionType,
        @Nonnull final LocalDate rEConditionValidityStartDate,
        @Nonnull final String rEExtConditionPurpose,
        @Nonnull final String rEStatusObjectCalculation )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("InternalRealEstateNumber", internalRealEstateNumber);
        key.put("RETermNumber", rETermNumber);
        key.put("ValidityStartEndDateValue", validityStartEndDateValue);
        key.put("REConditionType", rEConditionType);
        key.put("REConditionValidityStartDate", rEConditionValidityStartDate);
        key.put("REExtConditionPurpose", rEExtConditionPurpose);
        key.put("REStatusObjectCalculation", rEStatusObjectCalculation);
        return new GetByKeyRequestBuilder<REContrValuationCondition>(
            servicePath,
            REContrValuationCondition.class,
            key,
            "A_REContrValuationCondition");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<REContrValuationCondition> createREContrValuationCondition(
        @Nonnull final REContrValuationCondition rEContrValuationCondition )
    {
        return new CreateRequestBuilder<REContrValuationCondition>(
            servicePath,
            rEContrValuationCondition,
            "A_REContrValuationCondition");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<REContrValuationCondition> updateREContrValuationCondition(
        @Nonnull final REContrValuationCondition rEContrValuationCondition )
    {
        return new UpdateRequestBuilder<REContrValuationCondition>(
            servicePath,
            rEContrValuationCondition,
            "A_REContrValuationCondition");
    }

    @Override
    @Nonnull
    public DeleteRequestBuilder<REContrValuationCondition> deleteREContrValuationCondition(
        @Nonnull final REContrValuationCondition rEContrValuationCondition )
    {
        return new DeleteRequestBuilder<REContrValuationCondition>(
            servicePath,
            rEContrValuationCondition,
            "A_REContrValuationCondition");
    }

}
