/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.responsibilitymanagementteam.TeamAttributes;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.responsibilitymanagementteam.TeamCategories;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.responsibilitymanagementteam.TeamCategoryTexts;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.responsibilitymanagementteam.TeamHeaders;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.responsibilitymanagementteam.TeamMemberFunctions;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.responsibilitymanagementteam.TeamMembers;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.responsibilitymanagementteam.TeamOwners;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.responsibilitymanagementteam.TeamStatusTexts;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.responsibilitymanagementteam.TeamTypeTexts;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.responsibilitymanagementteam.TeamTypes;

import lombok.Getter;

/**
 * <p>
 * 
 * You can read team headers, team owners, responsibility definitions, team members, and the functions assigned to team
 * members. Each of these entities supports filtering of their properties and the properties of their associations. If
 * you have any problems trying to read these entities because of authorizations or the given filter criteria, the
 * system displays error messages in the response..
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/ce/db8a76b15e49c29b1d79d421e4b57c/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/API_RESPYM_TEAM_SRV_0001?section=OVERVIEW'>SAP
 * API Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>API_RESPYM_TEAM_SRV_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Responsibility Management Team Integration (SAP_COM_0560)</td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_RESPYMGMT_TEAM_SRV</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>ResponsibilityManagementTeam</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultResponsibilityManagementTeamService" )
public class DefaultResponsibilityManagementTeamService
    implements
    ServiceWithNavigableEntities,
    ResponsibilityManagementTeamService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link ResponsibilityManagementTeamService#DEFAULT_SERVICE_PATH} to send the requests.
     * 
     */
    public DefaultResponsibilityManagementTeamService()
    {
        servicePath = ResponsibilityManagementTeamService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultResponsibilityManagementTeamService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultResponsibilityManagementTeamService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultResponsibilityManagementTeamService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<TeamAttributes> getAllTeamAttributes()
    {
        return new GetAllRequestBuilder<TeamAttributes>(servicePath, TeamAttributes.class, "TeamAttributes");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<TeamAttributes> countTeamAttributes()
    {
        return new CountRequestBuilder<TeamAttributes>(servicePath, TeamAttributes.class, "TeamAttributes");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<TeamAttributes> getTeamAttributesByKey(
        @Nonnull final String respyMgmtGlobalTeamID,
        @Nonnull final String respyMgmtAttributeName,
        @Nonnull final String respyMgmtAttributeSign,
        @Nonnull final String respyMgmtAttributeOption,
        @Nonnull final String respyMgmtAttributeLowValue,
        @Nonnull final String respyMgmtAttributeHighValue )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("RespyMgmtGlobalTeamID", respyMgmtGlobalTeamID);
        key.put("RespyMgmtAttributeName", respyMgmtAttributeName);
        key.put("RespyMgmtAttributeSign", respyMgmtAttributeSign);
        key.put("RespyMgmtAttributeOption", respyMgmtAttributeOption);
        key.put("RespyMgmtAttributeLowValue", respyMgmtAttributeLowValue);
        key.put("RespyMgmtAttributeHighValue", respyMgmtAttributeHighValue);
        return new GetByKeyRequestBuilder<TeamAttributes>(servicePath, TeamAttributes.class, key, "TeamAttributes");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<TeamCategories> getAllTeamCategories()
    {
        return new GetAllRequestBuilder<TeamCategories>(servicePath, TeamCategories.class, "TeamCategories");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<TeamCategories> countTeamCategories()
    {
        return new CountRequestBuilder<TeamCategories>(servicePath, TeamCategories.class, "TeamCategories");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<TeamCategories> getTeamCategoriesByKey( @Nonnull final String respyMgmtTeamCategory )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("RespyMgmtTeamCategory", respyMgmtTeamCategory);
        return new GetByKeyRequestBuilder<TeamCategories>(servicePath, TeamCategories.class, key, "TeamCategories");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<TeamCategoryTexts> getAllTeamCategoryTexts()
    {
        return new GetAllRequestBuilder<TeamCategoryTexts>(servicePath, TeamCategoryTexts.class, "TeamCategoryTexts");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<TeamCategoryTexts> countTeamCategoryTexts()
    {
        return new CountRequestBuilder<TeamCategoryTexts>(servicePath, TeamCategoryTexts.class, "TeamCategoryTexts");
    }

    @Override
    @Nonnull
    public
        GetByKeyRequestBuilder<TeamCategoryTexts>
        getTeamCategoryTextsByKey( @Nonnull final String respyMgmtTeamCategory, @Nonnull final String language )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("RespyMgmtTeamCategory", respyMgmtTeamCategory);
        key.put("Language", language);
        return new GetByKeyRequestBuilder<TeamCategoryTexts>(
            servicePath,
            TeamCategoryTexts.class,
            key,
            "TeamCategoryTexts");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<TeamHeaders> getAllTeamHeaders()
    {
        return new GetAllRequestBuilder<TeamHeaders>(servicePath, TeamHeaders.class, "TeamHeaders");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<TeamHeaders> countTeamHeaders()
    {
        return new CountRequestBuilder<TeamHeaders>(servicePath, TeamHeaders.class, "TeamHeaders");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<TeamHeaders> getTeamHeadersByKey( @Nonnull final String respyMgmtGlobalTeamID )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("RespyMgmtGlobalTeamID", respyMgmtGlobalTeamID);
        return new GetByKeyRequestBuilder<TeamHeaders>(servicePath, TeamHeaders.class, key, "TeamHeaders");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<TeamMemberFunctions> getAllTeamMemberFunctions()
    {
        return new GetAllRequestBuilder<TeamMemberFunctions>(
            servicePath,
            TeamMemberFunctions.class,
            "TeamMemberFunctions");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<TeamMemberFunctions> countTeamMemberFunctions()
    {
        return new CountRequestBuilder<TeamMemberFunctions>(
            servicePath,
            TeamMemberFunctions.class,
            "TeamMemberFunctions");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<TeamMemberFunctions> getTeamMemberFunctionsByKey(
        @Nonnull final String respyMgmtGlobalTeamID,
        @Nonnull final String respyMgmtTeamMemberType,
        @Nonnull final String responsibilityTeamMember,
        @Nonnull final String respyMgmtFunction )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("RespyMgmtGlobalTeamID", respyMgmtGlobalTeamID);
        key.put("RespyMgmtTeamMemberType", respyMgmtTeamMemberType);
        key.put("ResponsibilityTeamMember", responsibilityTeamMember);
        key.put("RespyMgmtFunction", respyMgmtFunction);
        return new GetByKeyRequestBuilder<TeamMemberFunctions>(
            servicePath,
            TeamMemberFunctions.class,
            key,
            "TeamMemberFunctions");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<TeamMembers> getAllTeamMembers()
    {
        return new GetAllRequestBuilder<TeamMembers>(servicePath, TeamMembers.class, "TeamMembers");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<TeamMembers> countTeamMembers()
    {
        return new CountRequestBuilder<TeamMembers>(servicePath, TeamMembers.class, "TeamMembers");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<TeamMembers> getTeamMembersByKey(
        @Nonnull final String respyMgmtGlobalTeamID,
        @Nonnull final String respyMgmtTeamMemberType,
        @Nonnull final String responsibilityTeamMember )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("RespyMgmtGlobalTeamID", respyMgmtGlobalTeamID);
        key.put("RespyMgmtTeamMemberType", respyMgmtTeamMemberType);
        key.put("ResponsibilityTeamMember", responsibilityTeamMember);
        return new GetByKeyRequestBuilder<TeamMembers>(servicePath, TeamMembers.class, key, "TeamMembers");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<TeamOwners> getAllTeamOwners()
    {
        return new GetAllRequestBuilder<TeamOwners>(servicePath, TeamOwners.class, "TeamOwners");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<TeamOwners> countTeamOwners()
    {
        return new CountRequestBuilder<TeamOwners>(servicePath, TeamOwners.class, "TeamOwners");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<TeamOwners> getTeamOwnersByKey(
        @Nonnull final String respyMgmtGlobalTeamID,
        @Nonnull final String respyMgmtTeamOwnerType,
        @Nonnull final String responsibilityTeamOwner )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("RespyMgmtGlobalTeamID", respyMgmtGlobalTeamID);
        key.put("RespyMgmtTeamOwnerType", respyMgmtTeamOwnerType);
        key.put("ResponsibilityTeamOwner", responsibilityTeamOwner);
        return new GetByKeyRequestBuilder<TeamOwners>(servicePath, TeamOwners.class, key, "TeamOwners");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<TeamStatusTexts> getAllTeamStatusTexts()
    {
        return new GetAllRequestBuilder<TeamStatusTexts>(servicePath, TeamStatusTexts.class, "TeamStatusTexts");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<TeamStatusTexts> countTeamStatusTexts()
    {
        return new CountRequestBuilder<TeamStatusTexts>(servicePath, TeamStatusTexts.class, "TeamStatusTexts");
    }

    @Override
    @Nonnull
    public
        GetByKeyRequestBuilder<TeamStatusTexts>
        getTeamStatusTextsByKey( @Nonnull final String respyMgmtTeamStatus, @Nonnull final String language )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("RespyMgmtTeamStatus", respyMgmtTeamStatus);
        key.put("Language", language);
        return new GetByKeyRequestBuilder<TeamStatusTexts>(servicePath, TeamStatusTexts.class, key, "TeamStatusTexts");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<TeamTypes> getAllTeamTypes()
    {
        return new GetAllRequestBuilder<TeamTypes>(servicePath, TeamTypes.class, "TeamTypes");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<TeamTypes> countTeamTypes()
    {
        return new CountRequestBuilder<TeamTypes>(servicePath, TeamTypes.class, "TeamTypes");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<TeamTypes> getTeamTypesByKey( @Nonnull final String respyMgmtTeamType )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("RespyMgmtTeamType", respyMgmtTeamType);
        return new GetByKeyRequestBuilder<TeamTypes>(servicePath, TeamTypes.class, key, "TeamTypes");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<TeamTypeTexts> getAllTeamTypeTexts()
    {
        return new GetAllRequestBuilder<TeamTypeTexts>(servicePath, TeamTypeTexts.class, "TeamTypeTexts");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<TeamTypeTexts> countTeamTypeTexts()
    {
        return new CountRequestBuilder<TeamTypeTexts>(servicePath, TeamTypeTexts.class, "TeamTypeTexts");
    }

    @Override
    @Nonnull
    public
        GetByKeyRequestBuilder<TeamTypeTexts>
        getTeamTypeTextsByKey( @Nonnull final String respyMgmtTeamType, @Nonnull final String language )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("RespyMgmtTeamType", respyMgmtTeamType);
        key.put("Language", language);
        return new GetByKeyRequestBuilder<TeamTypeTexts>(servicePath, TeamTypeTexts.class, key, "TeamTypeTexts");
    }

}
