/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.settlementdocument.CustItmPricingElement;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.settlementdocument.CustPricingElement;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.settlementdocument.SetlDoc;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.settlementdocument.SetlDocItm;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.settlementdocument.SuplrItmPricingElement;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.settlementdocument.SuplrPricingElement;

import lombok.Getter;

/**
 * <p>
 * 
 * You retrieve the following data using this OData based service:
 * 
 * You can read the Settlement Documents on header and item level.
 * 
 * You can release Settlement Documents which were blocked to accounting and where the application status was not set.
 * 
 * You can reverse Settlement Documents.
 * 
 * You can read the relevant pricing elements for the Settlement Document header and item level.
 * 
 * You can complete Settlement Documents.
 * 
 * You can revoke the completion of Settlement Documents..
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/bf/855abdfc6045c0a7aa2536781917ec/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference:
 * <a href='https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/SETTLMTDOC_0001?section=OVERVIEW'>SAP
 * API Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>SETTLMTDOC_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Logistics - Settlement Document Integration (SAP_COM_0714)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/1B6'>Sales Rebate Processing (1B6)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/2R7'>Purchasing Rebate Processing (2R7)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/2TT'>Sales Commissions - External Sales Representative (2TT)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/55I'>Royalties Settlement for Sales (55I)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_SETTLMT_DOC</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>SettlementDocument</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultSettlementDocumentService" )
public class DefaultSettlementDocumentService implements ServiceWithNavigableEntities, SettlementDocumentService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link SettlementDocumentService#DEFAULT_SERVICE_PATH} to send the requests.
     * 
     */
    public DefaultSettlementDocumentService()
    {
        servicePath = SettlementDocumentService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultSettlementDocumentService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultSettlementDocumentService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultSettlementDocumentService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<CustItmPricingElement> getAllCustItmPricingElement()
    {
        return new GetAllRequestBuilder<CustItmPricingElement>(
            servicePath,
            CustItmPricingElement.class,
            "CustItmPricingElement");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<CustItmPricingElement> countCustItmPricingElement()
    {
        return new CountRequestBuilder<CustItmPricingElement>(
            servicePath,
            CustItmPricingElement.class,
            "CustItmPricingElement");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<CustItmPricingElement> getCustItmPricingElementByKey(
        @Nonnull final String settlmtDoc,
        @Nonnull final String settlmtDocItem,
        @Nonnull final String pricingProcedureStep,
        @Nonnull final String pricingProcedureCounter )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("SettlmtDoc", settlmtDoc);
        key.put("SettlmtDocItem", settlmtDocItem);
        key.put("PricingProcedureStep", pricingProcedureStep);
        key.put("PricingProcedureCounter", pricingProcedureCounter);
        return new GetByKeyRequestBuilder<CustItmPricingElement>(
            servicePath,
            CustItmPricingElement.class,
            key,
            "CustItmPricingElement");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<CustItmPricingElement> createCustItmPricingElement(
        @Nonnull final CustItmPricingElement custItmPricingElement )
    {
        return new CreateRequestBuilder<CustItmPricingElement>(
            servicePath,
            custItmPricingElement,
            "CustItmPricingElement");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<CustPricingElement> getAllCustPricingElement()
    {
        return new GetAllRequestBuilder<CustPricingElement>(
            servicePath,
            CustPricingElement.class,
            "CustPricingElement");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<CustPricingElement> countCustPricingElement()
    {
        return new CountRequestBuilder<CustPricingElement>(servicePath, CustPricingElement.class, "CustPricingElement");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<CustPricingElement> getCustPricingElementByKey(
        @Nonnull final String settlmtDoc,
        @Nonnull final String pricingProcedureStep,
        @Nonnull final String pricingProcedureCounter )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("SettlmtDoc", settlmtDoc);
        key.put("PricingProcedureStep", pricingProcedureStep);
        key.put("PricingProcedureCounter", pricingProcedureCounter);
        return new GetByKeyRequestBuilder<CustPricingElement>(
            servicePath,
            CustPricingElement.class,
            key,
            "CustPricingElement");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<CustPricingElement> createCustPricingElement(
        @Nonnull final CustPricingElement custPricingElement )
    {
        return new CreateRequestBuilder<CustPricingElement>(servicePath, custPricingElement, "CustPricingElement");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<SetlDoc> getAllSetlDoc()
    {
        return new GetAllRequestBuilder<SetlDoc>(servicePath, SetlDoc.class, "SetlDoc");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<SetlDoc> countSetlDoc()
    {
        return new CountRequestBuilder<SetlDoc>(servicePath, SetlDoc.class, "SetlDoc");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<SetlDoc> getSetlDocByKey( @Nonnull final String settlmtDoc )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("SettlmtDoc", settlmtDoc);
        return new GetByKeyRequestBuilder<SetlDoc>(servicePath, SetlDoc.class, key, "SetlDoc");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<SetlDoc> createSetlDoc( @Nonnull final SetlDoc setlDoc )
    {
        return new CreateRequestBuilder<SetlDoc>(servicePath, setlDoc, "SetlDoc");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<SetlDocItm> getAllSetlDocItm()
    {
        return new GetAllRequestBuilder<SetlDocItm>(servicePath, SetlDocItm.class, "SetlDocItm");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<SetlDocItm> countSetlDocItm()
    {
        return new CountRequestBuilder<SetlDocItm>(servicePath, SetlDocItm.class, "SetlDocItm");
    }

    @Override
    @Nonnull
    public
        GetByKeyRequestBuilder<SetlDocItm>
        getSetlDocItmByKey( @Nonnull final String settlmtDoc, @Nonnull final String settlmtDocItem )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("SettlmtDoc", settlmtDoc);
        key.put("SettlmtDocItem", settlmtDocItem);
        return new GetByKeyRequestBuilder<SetlDocItm>(servicePath, SetlDocItm.class, key, "SetlDocItm");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<SetlDocItm> createSetlDocItm( @Nonnull final SetlDocItm setlDocItm )
    {
        return new CreateRequestBuilder<SetlDocItm>(servicePath, setlDocItm, "SetlDocItm");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<SuplrItmPricingElement> getAllSuplrItmPricingElement()
    {
        return new GetAllRequestBuilder<SuplrItmPricingElement>(
            servicePath,
            SuplrItmPricingElement.class,
            "SuplrItmPricingElement");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<SuplrItmPricingElement> countSuplrItmPricingElement()
    {
        return new CountRequestBuilder<SuplrItmPricingElement>(
            servicePath,
            SuplrItmPricingElement.class,
            "SuplrItmPricingElement");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<SuplrItmPricingElement> getSuplrItmPricingElementByKey(
        @Nonnull final String settlmtDoc,
        @Nonnull final String settlmtDocItem,
        @Nonnull final String pricingProcedureStep,
        @Nonnull final String pricingProcedureCounter )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("SettlmtDoc", settlmtDoc);
        key.put("SettlmtDocItem", settlmtDocItem);
        key.put("PricingProcedureStep", pricingProcedureStep);
        key.put("PricingProcedureCounter", pricingProcedureCounter);
        return new GetByKeyRequestBuilder<SuplrItmPricingElement>(
            servicePath,
            SuplrItmPricingElement.class,
            key,
            "SuplrItmPricingElement");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<SuplrItmPricingElement> createSuplrItmPricingElement(
        @Nonnull final SuplrItmPricingElement suplrItmPricingElement )
    {
        return new CreateRequestBuilder<SuplrItmPricingElement>(
            servicePath,
            suplrItmPricingElement,
            "SuplrItmPricingElement");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<SuplrPricingElement> getAllSuplrPricingElement()
    {
        return new GetAllRequestBuilder<SuplrPricingElement>(
            servicePath,
            SuplrPricingElement.class,
            "SuplrPricingElement");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<SuplrPricingElement> countSuplrPricingElement()
    {
        return new CountRequestBuilder<SuplrPricingElement>(
            servicePath,
            SuplrPricingElement.class,
            "SuplrPricingElement");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<SuplrPricingElement> getSuplrPricingElementByKey(
        @Nonnull final String settlmtDoc,
        @Nonnull final String pricingProcedureStep,
        @Nonnull final String pricingProcedureCounter )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("SettlmtDoc", settlmtDoc);
        key.put("PricingProcedureStep", pricingProcedureStep);
        key.put("PricingProcedureCounter", pricingProcedureCounter);
        return new GetByKeyRequestBuilder<SuplrPricingElement>(
            servicePath,
            SuplrPricingElement.class,
            key,
            "SuplrPricingElement");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<SuplrPricingElement> createSuplrPricingElement(
        @Nonnull final SuplrPricingElement suplrPricingElement )
    {
        return new CreateRequestBuilder<SuplrPricingElement>(servicePath, suplrPricingElement, "SuplrPricingElement");
    }

}
