/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.settlementprocesstype.SettlmtApplStsGrpStsAssgmt;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.settlementprocesstype.SettlmtDocProcTypeAssgmt;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.settlementprocesstype.SettlmtDocType;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.settlementprocesstype.SettlmtProcessType;

import lombok.Getter;

/**
 * <p>
 * 
 * You retrieve the following data using this OData based service:
 * 
 * You can read the Settlement Process Types and related information.
 * 
 * You can navigate to the Application Status Assignment for a Settlement Process Type and read the assigned Application
 * Status Group and all the possible Application Statuses linked to it.
 * 
 * You can navigate to the Settlement Document Type Assignment for a Settlement Process Type and read the assigned
 * Settlement Document Types for the relevant Settlement Process Type. You can also read further information regarding
 * the Settlement Document Types..
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/7d/dd5319e58546d5b44a2f263eaf3e01/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/SETTLMTPROCESSTYPE_0001?section=OVERVIEW'>SAP
 * API Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>SETTLMTPROCESSTYPE_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Logistics - Settlement Process Type Integration (SAP_COM_0710)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/1B6'>Sales Rebate Processing (1B6)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/2R7'>Purchasing Rebate Processing (2R7)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/2TT'>Sales Commissions - External Sales Representative (2TT)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/3TD'>Internal Commissions Settlement with SAP Sales Cloud
 * (3TD)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_SETTLMT_PROC_TYPE</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>SettlementProcessType</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultSettlementProcessTypeService" )
public class DefaultSettlementProcessTypeService implements ServiceWithNavigableEntities, SettlementProcessTypeService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link SettlementProcessTypeService#DEFAULT_SERVICE_PATH} to send the requests.
     * 
     */
    public DefaultSettlementProcessTypeService()
    {
        servicePath = SettlementProcessTypeService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultSettlementProcessTypeService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultSettlementProcessTypeService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultSettlementProcessTypeService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<SettlmtApplStsGrpStsAssgmt> getAllSettlmtApplStsGrpStsAssgmt()
    {
        return new GetAllRequestBuilder<SettlmtApplStsGrpStsAssgmt>(
            servicePath,
            SettlmtApplStsGrpStsAssgmt.class,
            "SettlmtApplStsGrpStsAssgmt");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<SettlmtApplStsGrpStsAssgmt> countSettlmtApplStsGrpStsAssgmt()
    {
        return new CountRequestBuilder<SettlmtApplStsGrpStsAssgmt>(
            servicePath,
            SettlmtApplStsGrpStsAssgmt.class,
            "SettlmtApplStsGrpStsAssgmt");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<SettlmtApplStsGrpStsAssgmt> getSettlmtApplStsGrpStsAssgmtByKey(
        @Nonnull final String settlmtApplStsGrp,
        @Nonnull final String settlmtApplSts )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("SettlmtApplStsGrp", settlmtApplStsGrp);
        key.put("SettlmtApplSts", settlmtApplSts);
        return new GetByKeyRequestBuilder<SettlmtApplStsGrpStsAssgmt>(
            servicePath,
            SettlmtApplStsGrpStsAssgmt.class,
            key,
            "SettlmtApplStsGrpStsAssgmt");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<SettlmtDocProcTypeAssgmt> getAllSettlmtDocProcTypeAssgmt()
    {
        return new GetAllRequestBuilder<SettlmtDocProcTypeAssgmt>(
            servicePath,
            SettlmtDocProcTypeAssgmt.class,
            "SettlmtDocProcTypeAssgmt");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<SettlmtDocProcTypeAssgmt> countSettlmtDocProcTypeAssgmt()
    {
        return new CountRequestBuilder<SettlmtDocProcTypeAssgmt>(
            servicePath,
            SettlmtDocProcTypeAssgmt.class,
            "SettlmtDocProcTypeAssgmt");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<SettlmtDocProcTypeAssgmt> getSettlmtDocProcTypeAssgmtByKey(
        @Nonnull final String settlmtProcessType,
        @Nonnull final String settlmtDocType )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("SettlmtProcessType", settlmtProcessType);
        key.put("SettlmtDocType", settlmtDocType);
        return new GetByKeyRequestBuilder<SettlmtDocProcTypeAssgmt>(
            servicePath,
            SettlmtDocProcTypeAssgmt.class,
            key,
            "SettlmtDocProcTypeAssgmt");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<SettlmtDocType> getAllSettlmtDocType()
    {
        return new GetAllRequestBuilder<SettlmtDocType>(servicePath, SettlmtDocType.class, "SettlmtDocType");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<SettlmtDocType> countSettlmtDocType()
    {
        return new CountRequestBuilder<SettlmtDocType>(servicePath, SettlmtDocType.class, "SettlmtDocType");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<SettlmtDocType> getSettlmtDocTypeByKey( @Nonnull final String settlmtDocType )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("SettlmtDocType", settlmtDocType);
        return new GetByKeyRequestBuilder<SettlmtDocType>(servicePath, SettlmtDocType.class, key, "SettlmtDocType");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<SettlmtProcessType> getAllSettlmtProcessType()
    {
        return new GetAllRequestBuilder<SettlmtProcessType>(
            servicePath,
            SettlmtProcessType.class,
            "SettlmtProcessType");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<SettlmtProcessType> countSettlmtProcessType()
    {
        return new CountRequestBuilder<SettlmtProcessType>(servicePath, SettlmtProcessType.class, "SettlmtProcessType");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<SettlmtProcessType> getSettlmtProcessTypeByKey(
        @Nonnull final String settlmtProcessType )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("SettlmtProcessType", settlmtProcessType);
        return new GetByKeyRequestBuilder<SettlmtProcessType>(
            servicePath,
            SettlmtProcessType.class,
            key,
            "SettlmtProcessType");
    }

}
