/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.supplieritem.SupplierItem;

import lombok.Getter;

/**
 * <p>
 * You can use this service to create, read and update supplier item. The service is based on the OData protocol and can
 * be consumed by external systems.
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/74/c6b169bfb74953b07a66aa248cd9aa/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference:
 * <a href='https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/SUPPLIERITEM_0001?section=OVERVIEW'>SAP
 * API Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>SUPPLIERITEM_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Field Logistics - Manage Supplier Item Integration (SAP_COM_0732)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/4AH'>Field Logistics Planning and Execution – Supplier Items
 * (4AH)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_SUPPLIERITEM</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>FldLogsSupplierItem</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultSupplierItemService" )
public class DefaultSupplierItemService implements ServiceWithNavigableEntities, SupplierItemService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link SupplierItemService#DEFAULT_SERVICE_PATH} to send the requests.
     * 
     */
    public DefaultSupplierItemService()
    {
        servicePath = SupplierItemService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultSupplierItemService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultSupplierItemService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultSupplierItemService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<SupplierItem> getAllSupplierItem()
    {
        return new GetAllRequestBuilder<SupplierItem>(servicePath, SupplierItem.class, "SupplierItem");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<SupplierItem> countSupplierItem()
    {
        return new CountRequestBuilder<SupplierItem>(servicePath, SupplierItem.class, "SupplierItem");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<SupplierItem> getSupplierItemByKey( @Nonnull final UUID fldLogsSuplrItemUUID )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("FldLogsSuplrItemUUID", fldLogsSuplrItemUUID);
        return new GetByKeyRequestBuilder<SupplierItem>(servicePath, SupplierItem.class, key, "SupplierItem");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<SupplierItem> createSupplierItem( @Nonnull final SupplierItem supplierItem )
    {
        return new CreateRequestBuilder<SupplierItem>(servicePath, supplierItem, "SupplierItem");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<SupplierItem> updateSupplierItem( @Nonnull final SupplierItem supplierItem )
    {
        return new UpdateRequestBuilder<SupplierItem>(servicePath, supplierItem, "SupplierItem");
    }

}
