/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.DeleteRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.supplyavailabilityprotectionplan.SupProtCharacteristic;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.supplyavailabilityprotectionplan.SupplyProtection;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.supplyavailabilityprotectionplan.SupplyProtectionGroup;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.supplyavailabilityprotectionplan.SupplyProtectionTimeBucket;

import lombok.Getter;

/**
 * <p>
 * 
 * The service contains the supply protection object, the characteristics, the supply protection groups and the time
 * buckets. Once the supply availability protection plan has been created, the response contains the created object with
 * comprehensive information including UUIDs of the objects as well as error messages for the case that the creation was
 * not successful..
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/ce/d8d281241040358bc54ba57baae01d/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/API_SUPAVAILYPROTPLAN_0001?section=OVERVIEW'>SAP
 * API Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>API_SUPAVAILYPROTPLAN_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Supply Availability Protection Plan Integration (SAP_COM_0384)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/1JW'>Advanced Available-to-Promise Processing (1JW)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509, OAuth2</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_SUPAVAILYPROTPLAN</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>SupAvailabilityProtectionPlan</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultSupplyAvailabilityProtectionPlanService" )
public class DefaultSupplyAvailabilityProtectionPlanService
    implements
    ServiceWithNavigableEntities,
    SupplyAvailabilityProtectionPlanService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link SupplyAvailabilityProtectionPlanService#DEFAULT_SERVICE_PATH} to send the
     * requests.
     * 
     */
    public DefaultSupplyAvailabilityProtectionPlanService()
    {
        servicePath = SupplyAvailabilityProtectionPlanService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultSupplyAvailabilityProtectionPlanService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultSupplyAvailabilityProtectionPlanService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultSupplyAvailabilityProtectionPlanService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<SupplyProtection> getAllSupplyProtection()
    {
        return new GetAllRequestBuilder<SupplyProtection>(servicePath, SupplyProtection.class, "A_SupplyProtection");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<SupplyProtection> countSupplyProtection()
    {
        return new CountRequestBuilder<SupplyProtection>(servicePath, SupplyProtection.class, "A_SupplyProtection");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<SupplyProtection> getSupplyProtectionByKey( @Nonnull final UUID supplyProtectionUUID )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("SupplyProtectionUUID", supplyProtectionUUID);
        return new GetByKeyRequestBuilder<SupplyProtection>(
            servicePath,
            SupplyProtection.class,
            key,
            "A_SupplyProtection");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<SupplyProtection> createSupplyProtection(
        @Nonnull final SupplyProtection supplyProtection )
    {
        return new CreateRequestBuilder<SupplyProtection>(servicePath, supplyProtection, "A_SupplyProtection");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<SupplyProtection> updateSupplyProtection(
        @Nonnull final SupplyProtection supplyProtection )
    {
        return new UpdateRequestBuilder<SupplyProtection>(servicePath, supplyProtection, "A_SupplyProtection");
    }

    @Override
    @Nonnull
    public DeleteRequestBuilder<SupplyProtection> deleteSupplyProtection(
        @Nonnull final SupplyProtection supplyProtection )
    {
        return new DeleteRequestBuilder<SupplyProtection>(servicePath, supplyProtection, "A_SupplyProtection");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<SupplyProtectionGroup> getAllSupplyProtectionGroup()
    {
        return new GetAllRequestBuilder<SupplyProtectionGroup>(
            servicePath,
            SupplyProtectionGroup.class,
            "A_SupplyProtectionGroup");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<SupplyProtectionGroup> countSupplyProtectionGroup()
    {
        return new CountRequestBuilder<SupplyProtectionGroup>(
            servicePath,
            SupplyProtectionGroup.class,
            "A_SupplyProtectionGroup");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<SupplyProtectionGroup> getSupplyProtectionGroupByKey(
        @Nonnull final UUID supplyProtectionGroupUUID )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("SupplyProtectionGroupUUID", supplyProtectionGroupUUID);
        return new GetByKeyRequestBuilder<SupplyProtectionGroup>(
            servicePath,
            SupplyProtectionGroup.class,
            key,
            "A_SupplyProtectionGroup");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<SupplyProtectionGroup> updateSupplyProtectionGroup(
        @Nonnull final SupplyProtectionGroup supplyProtectionGroup )
    {
        return new UpdateRequestBuilder<SupplyProtectionGroup>(
            servicePath,
            supplyProtectionGroup,
            "A_SupplyProtectionGroup");
    }

    @Override
    @Nonnull
    public DeleteRequestBuilder<SupplyProtectionGroup> deleteSupplyProtectionGroup(
        @Nonnull final SupplyProtectionGroup supplyProtectionGroup )
    {
        return new DeleteRequestBuilder<SupplyProtectionGroup>(
            servicePath,
            supplyProtectionGroup,
            "A_SupplyProtectionGroup");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<SupplyProtectionTimeBucket> getAllSupplyProtectionTimeBucket()
    {
        return new GetAllRequestBuilder<SupplyProtectionTimeBucket>(
            servicePath,
            SupplyProtectionTimeBucket.class,
            "A_SupplyProtectionTimeBucket");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<SupplyProtectionTimeBucket> countSupplyProtectionTimeBucket()
    {
        return new CountRequestBuilder<SupplyProtectionTimeBucket>(
            servicePath,
            SupplyProtectionTimeBucket.class,
            "A_SupplyProtectionTimeBucket");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<SupplyProtectionTimeBucket> getSupplyProtectionTimeBucketByKey(
        @Nonnull final UUID supProtTimeBucketUUID )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("SupProtTimeBucketUUID", supProtTimeBucketUUID);
        return new GetByKeyRequestBuilder<SupplyProtectionTimeBucket>(
            servicePath,
            SupplyProtectionTimeBucket.class,
            key,
            "A_SupplyProtectionTimeBucket");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<SupplyProtectionTimeBucket> updateSupplyProtectionTimeBucket(
        @Nonnull final SupplyProtectionTimeBucket supplyProtectionTimeBucket )
    {
        return new UpdateRequestBuilder<SupplyProtectionTimeBucket>(
            servicePath,
            supplyProtectionTimeBucket,
            "A_SupplyProtectionTimeBucket");
    }

    @Override
    @Nonnull
    public DeleteRequestBuilder<SupplyProtectionTimeBucket> deleteSupplyProtectionTimeBucket(
        @Nonnull final SupplyProtectionTimeBucket supplyProtectionTimeBucket )
    {
        return new DeleteRequestBuilder<SupplyProtectionTimeBucket>(
            servicePath,
            supplyProtectionTimeBucket,
            "A_SupplyProtectionTimeBucket");
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<SupProtCharacteristic> getAllSupProtCharacteristic()
    {
        return new GetAllRequestBuilder<SupProtCharacteristic>(
            servicePath,
            SupProtCharacteristic.class,
            "A_SupProtCharacteristic");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<SupProtCharacteristic> countSupProtCharacteristic()
    {
        return new CountRequestBuilder<SupProtCharacteristic>(
            servicePath,
            SupProtCharacteristic.class,
            "A_SupProtCharacteristic");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<SupProtCharacteristic> getSupProtCharacteristicByKey(
        @Nonnull final UUID supProtCharacteristicUUID )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("SupProtCharacteristicUUID", supProtCharacteristicUUID);
        return new GetByKeyRequestBuilder<SupProtCharacteristic>(
            servicePath,
            SupProtCharacteristic.class,
            key,
            "A_SupProtCharacteristic");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<SupProtCharacteristic> updateSupProtCharacteristic(
        @Nonnull final SupProtCharacteristic supProtCharacteristic )
    {
        return new UpdateRequestBuilder<SupProtCharacteristic>(
            servicePath,
            supProtCharacteristic,
            "A_SupProtCharacteristic");
    }

    @Override
    @Nonnull
    public DeleteRequestBuilder<SupProtCharacteristic> deleteSupProtCharacteristic(
        @Nonnull final SupProtCharacteristic supProtCharacteristic )
    {
        return new DeleteRequestBuilder<SupProtCharacteristic>(
            servicePath,
            supProtCharacteristic,
            "A_SupProtCharacteristic");
    }

}
