/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.time.LocalDate;
import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.inject.Named;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.ServiceWithNavigableEntities;
import com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.timedependentstocklevel.ProdTimeDepdntStockLvl;

import lombok.Getter;

/**
 * <p>
 * 
 * You can create, read and update the time dependent stock levels such as max stock, reorder point and safety stock.
 * You can also split and merge the stock validity intervals..
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/dd/c1e3b71636456b898eff6b8c6abd8a/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/PRODTIMEDPDNTSTCK_0001?section=OVERVIEW'>SAP
 * API Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>PRODTIMEDPDNTSTCK_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Buffer Sizing Integration (SAP_COM_0359)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/1Y2'>Demand-Driven Buffer Level Management (1Y2)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_PROD_TIMEDPDNTSTCK_SRV</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>Product</td>
 * </tr>
 * </table>
 * 
 */
@Named( "com.sap.cloud.sdk.s4hana.datamodel.odatav4.services.DefaultTimeDependentStockLevelService" )
public class DefaultTimeDependentStockLevelService
    implements
    ServiceWithNavigableEntities,
    TimeDependentStockLevelService
{

    @Nonnull
    @Getter
    private final String servicePath;

    /**
     * Creates a service using {@link TimeDependentStockLevelService#DEFAULT_SERVICE_PATH} to send the requests.
     * 
     */
    public DefaultTimeDependentStockLevelService()
    {
        servicePath = TimeDependentStockLevelService.DEFAULT_SERVICE_PATH;
    }

    /**
     * Creates a service using the provided service path to send the requests.
     * <p>
     * Used by the fluent {@link #withServicePath(String)} method.
     * 
     */
    private DefaultTimeDependentStockLevelService( @Nonnull final String servicePath )
    {
        this.servicePath = servicePath;
    }

    @Override
    @Nonnull
    public DefaultTimeDependentStockLevelService withServicePath( @Nonnull final String servicePath )
    {
        return new DefaultTimeDependentStockLevelService(servicePath);
    }

    @Override
    @Nonnull
    public BatchRequestBuilder batch()
    {
        return new BatchRequestBuilder(servicePath);
    }

    @Override
    @Nonnull
    public GetAllRequestBuilder<ProdTimeDepdntStockLvl> getAllProdTimeDepdntStockLvl()
    {
        return new GetAllRequestBuilder<ProdTimeDepdntStockLvl>(
            servicePath,
            ProdTimeDepdntStockLvl.class,
            "A_ProdTimeDepdntStockLvl");
    }

    @Override
    @Nonnull
    public CountRequestBuilder<ProdTimeDepdntStockLvl> countProdTimeDepdntStockLvl()
    {
        return new CountRequestBuilder<ProdTimeDepdntStockLvl>(
            servicePath,
            ProdTimeDepdntStockLvl.class,
            "A_ProdTimeDepdntStockLvl");
    }

    @Override
    @Nonnull
    public GetByKeyRequestBuilder<ProdTimeDepdntStockLvl> getProdTimeDepdntStockLvlByKey(
        @Nonnull final String product,
        @Nonnull final String plant,
        @Nonnull final String mRPArea,
        @Nonnull final LocalDate stockLevelValidityStartDate )
    {
        final Map<String, Object> key = new HashMap<String, Object>();
        key.put("Product", product);
        key.put("Plant", plant);
        key.put("MRPArea", mRPArea);
        key.put("StockLevelValidityStartDate", stockLevelValidityStartDate);
        return new GetByKeyRequestBuilder<ProdTimeDepdntStockLvl>(
            servicePath,
            ProdTimeDepdntStockLvl.class,
            key,
            "A_ProdTimeDepdntStockLvl");
    }

    @Override
    @Nonnull
    public CreateRequestBuilder<ProdTimeDepdntStockLvl> createProdTimeDepdntStockLvl(
        @Nonnull final ProdTimeDepdntStockLvl prodTimeDepdntStockLvl )
    {
        return new CreateRequestBuilder<ProdTimeDepdntStockLvl>(
            servicePath,
            prodTimeDepdntStockLvl,
            "A_ProdTimeDepdntStockLvl");
    }

    @Override
    @Nonnull
    public UpdateRequestBuilder<ProdTimeDepdntStockLvl> updateProdTimeDepdntStockLvl(
        @Nonnull final ProdTimeDepdntStockLvl prodTimeDepdntStockLvl )
    {
        return new UpdateRequestBuilder<ProdTimeDepdntStockLvl>(
            servicePath,
            prodTimeDepdntStockLvl,
            "A_ProdTimeDepdntStockLvl");
    }

}
