/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.util.UUID;

import javax.annotation.Nonnull;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.handlingunit.HandlingUnit;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.handlingunit.HandlingUnitItem;

/**
 * <p>
 * 
 * The service contains handling unit header and item nodes. With this API, you can carry out the following operations:
 * - Retrieve header and item information from handling units, for example, displaying all handling for a plant, or
 * displaying the products packed in a specific handling unit. For handling units processed in Warehouse Management,
 * warehouse number and handling unit number must be used as a key. For handling units not relevant for Warehouse
 * Management, the handling unit number is used as a key. - Create handling units without reference to a document. These
 * non-assigned handling units are used, for example, when the physical creation of a handling unit is planned or when a
 * handling unit already exists physically but is not yet entered in the system. - Post goods movement for handling
 * units. You can move handling units to different plants and to different storage locations within one plant and
 * transfer handling units from an SAP S/4HANA storage location to a Warehouse Management storage bin. .
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/08/31a7112057480a8f61b8bf01b56202/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference:
 * <a href='https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/HANDLINGUNIT_0001?section=OVERVIEW'>SAP
 * API Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>HANDLINGUNIT_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Handling Unit Integration (SAP_COM_0364)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/3BS'>Warehouse Outbound Processing (3BS)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BD9'>Sell from Stock (BD9)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_HANDLINGUNIT</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>HandlingUnit</td>
 * </tr>
 * </table>
 * 
 */
public interface HandlingUnitService
{

    /**
     * If no other path was provided via the {@link #withServicePath(String)} method, this is the default service path
     * used to access the endpoint.
     * 
     */
    String DEFAULT_SERVICE_PATH = "/sap/opu/odata4/sap/api_handlingunit/srvd_a2x/sap/handlingunit/0001";

    /**
     * Overrides the default service path and returns a new service instance with the specified service path. Also
     * adjusts the respective entity URLs.
     * 
     * @param servicePath
     *            Service path that will override the default.
     * @return A new service instance with the specified service path.
     */
    @Nonnull
    HandlingUnitService withServicePath( @Nonnull final String servicePath );

    /**
     * Creates a batch request builder object.
     * 
     * @return A request builder to handle batch operation on this service. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder#execute(HttpDestinationProperties)
     *         execute} method on the request builder object.
     */
    @Nonnull
    BatchRequestBuilder batch();

    /**
     * Fetch multiple {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.handlingunit.HandlingUnit
     * HandlingUnit} entities.
     * 
     * @return A request builder to fetch multiple
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.handlingunit.HandlingUnit HandlingUnit}
     *         entities. This request builder allows methods which modify the underlying query to be called before
     *         executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.handlingunit.HandlingUnit>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetAllRequestBuilder<HandlingUnit> getAllHandlingUnit();

    /**
     * Fetch the number of entries from the
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.handlingunit.HandlingUnit HandlingUnit} entity
     * collection matching the filter and search expressions.
     * 
     * @return A request builder to fetch the count of
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.handlingunit.HandlingUnit HandlingUnit}
     *         entities. This request builder allows methods which modify the underlying query to be called before
     *         executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.handlingunit.HandlingUnit>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CountRequestBuilder<HandlingUnit> countHandlingUnit();

    /**
     * Fetch a single {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.handlingunit.HandlingUnit
     * HandlingUnit} entity using key fields.
     * 
     * @param handlingUnitExternalID
     *            External Handling Unit Identification
     *            <p>
     *            Constraints: Not nullable, Maximum length: 20
     *            </p>
     * @param warehouse
     *            Warehouse Number/Warehouse Complex
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @return A request builder to fetch a single
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.handlingunit.HandlingUnit HandlingUnit}
     *         entity using key fields. This request builder allows methods which modify the underlying query to be
     *         called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.handlingunit.HandlingUnit>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetByKeyRequestBuilder<HandlingUnit>
        getHandlingUnitByKey( @Nonnull final String handlingUnitExternalID, @Nonnull final String warehouse );

    /**
     * Create a new {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.handlingunit.HandlingUnit HandlingUnit}
     * entity and save it to the S/4HANA system.
     * 
     * @param handlingUnit
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.handlingunit.HandlingUnit HandlingUnit}
     *            entity object that will be created in the S/4HANA system.
     * @return A request builder to create a new
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.handlingunit.HandlingUnit HandlingUnit}
     *         entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.handlingunit.HandlingUnit>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CreateRequestBuilder<HandlingUnit> createHandlingUnit( @Nonnull final HandlingUnit handlingUnit );

    /**
     * Fetch multiple {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.handlingunit.HandlingUnitItem
     * HandlingUnitItem} entities.
     * 
     * @return A request builder to fetch multiple
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.handlingunit.HandlingUnitItem
     *         HandlingUnitItem} entities. This request builder allows methods which modify the underlying query to be
     *         called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.handlingunit.HandlingUnitItem>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetAllRequestBuilder<HandlingUnitItem> getAllHandlingUnitItem();

    /**
     * Fetch the number of entries from the
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.handlingunit.HandlingUnitItem HandlingUnitItem}
     * entity collection matching the filter and search expressions.
     * 
     * @return A request builder to fetch the count of
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.handlingunit.HandlingUnitItem
     *         HandlingUnitItem} entities. This request builder allows methods which modify the underlying query to be
     *         called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.handlingunit.HandlingUnitItem>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CountRequestBuilder<HandlingUnitItem> countHandlingUnitItem();

    /**
     * Fetch a single {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.handlingunit.HandlingUnitItem
     * HandlingUnitItem} entity using key fields.
     * 
     * @param handlingUnitExternalID
     *            External Handling Unit Identification
     *            <p>
     *            Constraints: Not nullable, Maximum length: 20
     *            </p>
     * @param stockItemUUID
     *            GUID Stock Item
     *            <p>
     *            Constraints: Not nullable
     *            </p>
     * @param warehouse
     *            Warehouse Number/Warehouse Complex
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @return A request builder to fetch a single
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.handlingunit.HandlingUnitItem
     *         HandlingUnitItem} entity using key fields. This request builder allows methods which modify the
     *         underlying query to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.handlingunit.HandlingUnitItem>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetByKeyRequestBuilder<HandlingUnitItem> getHandlingUnitItemByKey(
        @Nonnull final String handlingUnitExternalID,
        @Nonnull final String warehouse,
        @Nonnull final UUID stockItemUUID );

    /**
     * Create a new {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.handlingunit.HandlingUnitItem
     * HandlingUnitItem} entity and save it to the S/4HANA system.
     * 
     * @param handlingUnitItem
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.handlingunit.HandlingUnitItem
     *            HandlingUnitItem} entity object that will be created in the S/4HANA system.
     * @return A request builder to create a new
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.handlingunit.HandlingUnitItem
     *         HandlingUnitItem} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.handlingunit.HandlingUnitItem>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CreateRequestBuilder<HandlingUnitItem> createHandlingUnitItem( @Nonnull final HandlingUnitItem handlingUnitItem );

}
