/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import javax.annotation.Nonnull;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JntVntrPartEquityShr;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVenture;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureDescription;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityGroup;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityType;

/**
 * <p>
 * The service contains Joint Venture, Description, Equity Group, Equity Type, and Equity Share nodes. The information
 * is sent in the request as a payload, which is used in creating or updating a joint venture, including appropriate
 * checks for restricted values and authorizations. If there are any issues when the joint venture is created or
 * updated, the system displays error messages in the response. The service does not allow deletion of any of joint
 * ventures, descriptions, equity group, equity type or equity share.
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/a9/bbf241929d4abe8e685c877cc26428/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/API_JOINTVENTURE_0001?section=OVERVIEW'>SAP API
 * Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>API_JOINTVENTURE_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Joint Venture Accounting API Integration (SAP_COM_0383)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/3F7'>Joint Venture Accounting (3F7)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_JOINTVENTURE</td>
 * </tr>
 * </table>
 * 
 */
public interface JointVentureService
{

    /**
     * If no other path was provided via the {@link #withServicePath(String)} method, this is the default service path
     * used to access the endpoint.
     * 
     */
    String DEFAULT_SERVICE_PATH = "/sap/opu/odata4/sap/api_jointventure/srvd_a2x/sap/api_jointventure/0001";

    /**
     * Overrides the default service path and returns a new service instance with the specified service path. Also
     * adjusts the respective entity URLs.
     * 
     * @param servicePath
     *            Service path that will override the default.
     * @return A new service instance with the specified service path.
     */
    @Nonnull
    JointVentureService withServicePath( @Nonnull final String servicePath );

    /**
     * Creates a batch request builder object.
     * 
     * @return A request builder to handle batch operation on this service. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder#execute(HttpDestinationProperties)
     *         execute} method on the request builder object.
     */
    @Nonnull
    BatchRequestBuilder batch();

    /**
     * Fetch multiple {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JntVntrPartEquityShr
     * JntVntrPartEquityShr} entities.
     * 
     * @return A request builder to fetch multiple
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JntVntrPartEquityShr
     *         JntVntrPartEquityShr} entities. This request builder allows methods which modify the underlying query to
     *         be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JntVntrPartEquityShr>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetAllRequestBuilder<JntVntrPartEquityShr> getAllJntVntrPartEquityShr();

    /**
     * Fetch the number of entries from the
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JntVntrPartEquityShr
     * JntVntrPartEquityShr} entity collection matching the filter and search expressions.
     * 
     * @return A request builder to fetch the count of
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JntVntrPartEquityShr
     *         JntVntrPartEquityShr} entities. This request builder allows methods which modify the underlying query to
     *         be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JntVntrPartEquityShr>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CountRequestBuilder<JntVntrPartEquityShr> countJntVntrPartEquityShr();

    /**
     * Fetch a single {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JntVntrPartEquityShr
     * JntVntrPartEquityShr} entity using key fields.
     * 
     * @param companyCode
     *            Company Code
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param jointVenture
     *            Joint Venture
     *            <p>
     *            Constraints: Not nullable, Maximum length: 6
     *            </p>
     * @param jntOpgAgrmtEquityGrp
     *            Equity Group
     *            <p>
     *            Constraints: Not nullable, Maximum length: 3
     *            </p>
     * @param jointVenturePartner
     *            JV partner
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @return A request builder to fetch a single
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JntVntrPartEquityShr
     *         JntVntrPartEquityShr} entity using key fields. This request builder allows methods which modify the
     *         underlying query to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JntVntrPartEquityShr>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetByKeyRequestBuilder<JntVntrPartEquityShr> getJntVntrPartEquityShrByKey(
        @Nonnull final String companyCode,
        @Nonnull final String jointVenture,
        @Nonnull final String jntOpgAgrmtEquityGrp,
        @Nonnull final String jointVenturePartner );

    /**
     * Create a new {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JntVntrPartEquityShr
     * JntVntrPartEquityShr} entity and save it to the S/4HANA system.
     * 
     * @param jntVntrPartEquityShr
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JntVntrPartEquityShr
     *            JntVntrPartEquityShr} entity object that will be created in the S/4HANA system.
     * @return A request builder to create a new
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JntVntrPartEquityShr
     *         JntVntrPartEquityShr} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JntVntrPartEquityShr>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CreateRequestBuilder<JntVntrPartEquityShr>
        createJntVntrPartEquityShr( @Nonnull final JntVntrPartEquityShr jntVntrPartEquityShr );

    /**
     * Update an existing {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JntVntrPartEquityShr
     * JntVntrPartEquityShr} entity and save it to the S/4HANA system.
     * 
     * @param jntVntrPartEquityShr
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JntVntrPartEquityShr
     *            JntVntrPartEquityShr} entity object that will be updated in the S/4HANA system.
     * @return A request builder to update an existing
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JntVntrPartEquityShr
     *         JntVntrPartEquityShr} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JntVntrPartEquityShr>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    UpdateRequestBuilder<JntVntrPartEquityShr>
        updateJntVntrPartEquityShr( @Nonnull final JntVntrPartEquityShr jntVntrPartEquityShr );

    /**
     * Fetch multiple {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVenture
     * JointVenture} entities.
     * 
     * @return A request builder to fetch multiple
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVenture JointVenture}
     *         entities. This request builder allows methods which modify the underlying query to be called before
     *         executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVenture>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetAllRequestBuilder<JointVenture> getAllJointVenture();

    /**
     * Fetch the number of entries from the
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVenture JointVenture} entity
     * collection matching the filter and search expressions.
     * 
     * @return A request builder to fetch the count of
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVenture JointVenture}
     *         entities. This request builder allows methods which modify the underlying query to be called before
     *         executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVenture>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CountRequestBuilder<JointVenture> countJointVenture();

    /**
     * Fetch a single {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVenture
     * JointVenture} entity using key fields.
     * 
     * @param companyCode
     *            Company Code
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param jointVenture
     *            Joint Venture
     *            <p>
     *            Constraints: Not nullable, Maximum length: 6
     *            </p>
     * @return A request builder to fetch a single
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVenture JointVenture}
     *         entity using key fields. This request builder allows methods which modify the underlying query to be
     *         called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVenture>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetByKeyRequestBuilder<JointVenture>
        getJointVentureByKey( @Nonnull final String companyCode, @Nonnull final String jointVenture );

    /**
     * Create a new {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVenture JointVenture}
     * entity and save it to the S/4HANA system.
     * 
     * @param jointVenture
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVenture JointVenture}
     *            entity object that will be created in the S/4HANA system.
     * @return A request builder to create a new
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVenture JointVenture}
     *         entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVenture>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CreateRequestBuilder<JointVenture> createJointVenture( @Nonnull final JointVenture jointVenture );

    /**
     * Update an existing {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVenture
     * JointVenture} entity and save it to the S/4HANA system.
     * 
     * @param jointVenture
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVenture JointVenture}
     *            entity object that will be updated in the S/4HANA system.
     * @return A request builder to update an existing
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVenture JointVenture}
     *         entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVenture>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    UpdateRequestBuilder<JointVenture> updateJointVenture( @Nonnull final JointVenture jointVenture );

    /**
     * Fetch multiple {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureDescription
     * JointVentureDescription} entities.
     * 
     * @return A request builder to fetch multiple
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureDescription
     *         JointVentureDescription} entities. This request builder allows methods which modify the underlying query
     *         to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureDescription>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetAllRequestBuilder<JointVentureDescription> getAllJointVentureDescription();

    /**
     * Fetch the number of entries from the
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureDescription
     * JointVentureDescription} entity collection matching the filter and search expressions.
     * 
     * @return A request builder to fetch the count of
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureDescription
     *         JointVentureDescription} entities. This request builder allows methods which modify the underlying query
     *         to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureDescription>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CountRequestBuilder<JointVentureDescription> countJointVentureDescription();

    /**
     * Fetch a single {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureDescription
     * JointVentureDescription} entity using key fields.
     * 
     * @param companyCode
     *            Company Code
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param jointVenture
     *            Joint Venture
     *            <p>
     *            Constraints: Not nullable, Maximum length: 6
     *            </p>
     * @param language
     *            Language Key
     *            <p>
     *            Constraints: Not nullable, Maximum length: 2
     *            </p>
     * @return A request builder to fetch a single
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureDescription
     *         JointVentureDescription} entity using key fields. This request builder allows methods which modify the
     *         underlying query to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureDescription>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetByKeyRequestBuilder<JointVentureDescription> getJointVentureDescriptionByKey(
        @Nonnull final String companyCode,
        @Nonnull final String language,
        @Nonnull final String jointVenture );

    /**
     * Create a new {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureDescription
     * JointVentureDescription} entity and save it to the S/4HANA system.
     * 
     * @param jointVentureDescription
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureDescription
     *            JointVentureDescription} entity object that will be created in the S/4HANA system.
     * @return A request builder to create a new
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureDescription
     *         JointVentureDescription} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureDescription>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CreateRequestBuilder<JointVentureDescription>
        createJointVentureDescription( @Nonnull final JointVentureDescription jointVentureDescription );

    /**
     * Update an existing
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureDescription
     * JointVentureDescription} entity and save it to the S/4HANA system.
     * 
     * @param jointVentureDescription
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureDescription
     *            JointVentureDescription} entity object that will be updated in the S/4HANA system.
     * @return A request builder to update an existing
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureDescription
     *         JointVentureDescription} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureDescription>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    UpdateRequestBuilder<JointVentureDescription>
        updateJointVentureDescription( @Nonnull final JointVentureDescription jointVentureDescription );

    /**
     * Fetch multiple {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityGroup
     * JointVentureEquityGroup} entities.
     * 
     * @return A request builder to fetch multiple
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityGroup
     *         JointVentureEquityGroup} entities. This request builder allows methods which modify the underlying query
     *         to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityGroup>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetAllRequestBuilder<JointVentureEquityGroup> getAllJointVentureEquityGroup();

    /**
     * Fetch the number of entries from the
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityGroup
     * JointVentureEquityGroup} entity collection matching the filter and search expressions.
     * 
     * @return A request builder to fetch the count of
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityGroup
     *         JointVentureEquityGroup} entities. This request builder allows methods which modify the underlying query
     *         to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityGroup>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CountRequestBuilder<JointVentureEquityGroup> countJointVentureEquityGroup();

    /**
     * Fetch a single {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityGroup
     * JointVentureEquityGroup} entity using key fields.
     * 
     * @param companyCode
     *            Company Code
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param jointVenture
     *            Joint Venture
     *            <p>
     *            Constraints: Not nullable, Maximum length: 6
     *            </p>
     * @param jntOpgAgrmtEquityGrp
     *            Equity Group
     *            <p>
     *            Constraints: Not nullable, Maximum length: 3
     *            </p>
     * @return A request builder to fetch a single
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityGroup
     *         JointVentureEquityGroup} entity using key fields. This request builder allows methods which modify the
     *         underlying query to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityGroup>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetByKeyRequestBuilder<JointVentureEquityGroup> getJointVentureEquityGroupByKey(
        @Nonnull final String companyCode,
        @Nonnull final String jointVenture,
        @Nonnull final String jntOpgAgrmtEquityGrp );

    /**
     * Create a new {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityGroup
     * JointVentureEquityGroup} entity and save it to the S/4HANA system.
     * 
     * @param jointVentureEquityGroup
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityGroup
     *            JointVentureEquityGroup} entity object that will be created in the S/4HANA system.
     * @return A request builder to create a new
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityGroup
     *         JointVentureEquityGroup} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityGroup>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CreateRequestBuilder<JointVentureEquityGroup>
        createJointVentureEquityGroup( @Nonnull final JointVentureEquityGroup jointVentureEquityGroup );

    /**
     * Update an existing
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityGroup
     * JointVentureEquityGroup} entity and save it to the S/4HANA system.
     * 
     * @param jointVentureEquityGroup
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityGroup
     *            JointVentureEquityGroup} entity object that will be updated in the S/4HANA system.
     * @return A request builder to update an existing
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityGroup
     *         JointVentureEquityGroup} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityGroup>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    UpdateRequestBuilder<JointVentureEquityGroup>
        updateJointVentureEquityGroup( @Nonnull final JointVentureEquityGroup jointVentureEquityGroup );

    /**
     * Fetch multiple {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityType
     * JointVentureEquityType} entities.
     * 
     * @return A request builder to fetch multiple
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityType
     *         JointVentureEquityType} entities. This request builder allows methods which modify the underlying query
     *         to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityType>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetAllRequestBuilder<JointVentureEquityType> getAllJointVentureEquityType();

    /**
     * Fetch the number of entries from the
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityType
     * JointVentureEquityType} entity collection matching the filter and search expressions.
     * 
     * @return A request builder to fetch the count of
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityType
     *         JointVentureEquityType} entities. This request builder allows methods which modify the underlying query
     *         to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityType>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CountRequestBuilder<JointVentureEquityType> countJointVentureEquityType();

    /**
     * Fetch a single {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityType
     * JointVentureEquityType} entity using key fields.
     * 
     * @param companyCode
     *            Company Code
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param jointVentureEquityType
     *            Equity Type
     *            <p>
     *            Constraints: Not nullable, Maximum length: 3
     *            </p>
     * @param jointVenture
     *            Joint Venture
     *            <p>
     *            Constraints: Not nullable, Maximum length: 6
     *            </p>
     * @param jointVentureEquityFundingDate
     *            Valid from date (inverted: 99999999 - YYYYMMDD)
     *            <p>
     *            Constraints: Not nullable, Maximum length: 8
     *            </p>
     * @return A request builder to fetch a single
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityType
     *         JointVentureEquityType} entity using key fields. This request builder allows methods which modify the
     *         underlying query to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityType>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetByKeyRequestBuilder<JointVentureEquityType> getJointVentureEquityTypeByKey(
        @Nonnull final String companyCode,
        @Nonnull final String jointVenture,
        @Nonnull final String jointVentureEquityType,
        @Nonnull final String jointVentureEquityFundingDate );

    /**
     * Create a new {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityType
     * JointVentureEquityType} entity and save it to the S/4HANA system.
     * 
     * @param jointVentureEquityType
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityType
     *            JointVentureEquityType} entity object that will be created in the S/4HANA system.
     * @return A request builder to create a new
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityType
     *         JointVentureEquityType} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityType>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CreateRequestBuilder<JointVentureEquityType>
        createJointVentureEquityType( @Nonnull final JointVentureEquityType jointVentureEquityType );

    /**
     * Update an existing
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityType
     * JointVentureEquityType} entity and save it to the S/4HANA system.
     * 
     * @param jointVentureEquityType
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityType
     *            JointVentureEquityType} entity object that will be updated in the S/4HANA system.
     * @return A request builder to update an existing
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityType
     *         JointVentureEquityType} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.jointventure.JointVentureEquityType>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    UpdateRequestBuilder<JointVentureEquityType>
        updateJointVentureEquityType( @Nonnull final JointVentureEquityType jointVentureEquityType );

}
