/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.time.LocalDate;
import java.time.LocalTime;

import javax.annotation.Nonnull;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.FailedMeasurementReading;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MeasurementDocument;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MsmtDocumentLongText;

/**
 * <p>
 * 
 * The data transferred to the system after a measurement has been taken at a measuring point or a counter is described
 * in the SAP system as a measurement document.
 * 
 * The payload used to create a measurement document through this API is sent in JSON format as a request object. The
 * long text for the measurement document is included within as a nested object. .
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/6a/fa93607aa74b1bb0aebdbe2867c8f9/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference: <a href=
 * 'https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/MEASUREMENTDOCUMENT_0001?section=OVERVIEW'>SAP
 * API Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>MEASUREMENTDOCUMENT_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Plant Maintenance Execution Integration (SAP_COM_0398)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/3D2'>Service Order Management and Monitoring (3D2)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/3MO'>Service Contract Management (3MO)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/3XK'>In-House Repair (3XK)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/49X'>Service Order Processing with SAP Field Service Management
 * (49X)</a>, <a href='https://rapid.sap.com/bp/#/scopeitems/4GA'>Service Quotation (4GA)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/4HH'>Reactive Maintenance (4HH)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/4HI'>Proactive Maintenance (4HI)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/4VT'>Improvement Maintenance (4VT)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/4WM'>Operational and Overhead Maintenance (4WM)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BH1'>Corrective Maintenance (BH1)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BH2'>Emergency Maintenance (BH2)</a>,
 * <a href='https://rapid.sap.com/bp/#/scopeitems/BJ2'>Preventive Maintenance (BJ2)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509, OAuth2</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_MEASUREMENTDOCUMENT</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>MeasurementDocument</td>
 * </tr>
 * </table>
 * 
 */
public interface MeasurementDocumentService
{

    /**
     * If no other path was provided via the {@link #withServicePath(String)} method, this is the default service path
     * used to access the endpoint.
     * 
     */
    String DEFAULT_SERVICE_PATH = "/sap/opu/odata4/sap/api_measurementdocument/srvd_a2x/sap/measurementdocument/0001";

    /**
     * Overrides the default service path and returns a new service instance with the specified service path. Also
     * adjusts the respective entity URLs.
     * 
     * @param servicePath
     *            Service path that will override the default.
     * @return A new service instance with the specified service path.
     */
    @Nonnull
    MeasurementDocumentService withServicePath( @Nonnull final String servicePath );

    /**
     * Creates a batch request builder object.
     * 
     * @return A request builder to handle batch operation on this service. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder#execute(HttpDestinationProperties)
     *         execute} method on the request builder object.
     */
    @Nonnull
    BatchRequestBuilder batch();

    /**
     * Fetch multiple
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.FailedMeasurementReading
     * FailedMeasurementReading} entities.
     * 
     * @return A request builder to fetch multiple
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.FailedMeasurementReading
     *         FailedMeasurementReading} entities. This request builder allows methods which modify the underlying query
     *         to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.FailedMeasurementReading>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetAllRequestBuilder<FailedMeasurementReading> getAllFailedMeasurementReading();

    /**
     * Fetch the number of entries from the
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.FailedMeasurementReading
     * FailedMeasurementReading} entity collection matching the filter and search expressions.
     * 
     * @return A request builder to fetch the count of
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.FailedMeasurementReading
     *         FailedMeasurementReading} entities. This request builder allows methods which modify the underlying query
     *         to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.FailedMeasurementReading>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CountRequestBuilder<FailedMeasurementReading> countFailedMeasurementReading();

    /**
     * Fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.FailedMeasurementReading
     * FailedMeasurementReading} entity using key fields.
     * 
     * @param msmtRdngDate
     *            Date of the Measurement
     *            <p>
     *            Constraints: Not nullable
     *            </p>
     * @param measuringPoint
     *            Measuring Point
     *            <p>
     *            Constraints: Not nullable, Maximum length: 12
     *            </p>
     * @param msmtRdngTime
     *            Time of Measurement
     *            <p>
     *            Constraints: Not nullable
     *            </p>
     * @param msmtRdngSourceMeasurementDoc
     *            Source Document for Measurement Reading Transfer
     *            <p>
     *            Constraints: Not nullable, Maximum length: 20
     *            </p>
     * @return A request builder to fetch a single
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.FailedMeasurementReading
     *         FailedMeasurementReading} entity using key fields. This request builder allows methods which modify the
     *         underlying query to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.FailedMeasurementReading>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetByKeyRequestBuilder<FailedMeasurementReading> getFailedMeasurementReadingByKey(
        @Nonnull final LocalDate msmtRdngDate,
        @Nonnull final LocalTime msmtRdngTime,
        @Nonnull final String msmtRdngSourceMeasurementDoc,
        @Nonnull final String measuringPoint );

    /**
     * Create a new
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.FailedMeasurementReading
     * FailedMeasurementReading} entity and save it to the S/4HANA system.
     * 
     * @param failedMeasurementReading
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.FailedMeasurementReading
     *            FailedMeasurementReading} entity object that will be created in the S/4HANA system.
     * @return A request builder to create a new
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.FailedMeasurementReading
     *         FailedMeasurementReading} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.FailedMeasurementReading>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CreateRequestBuilder<FailedMeasurementReading>
        createFailedMeasurementReading( @Nonnull final FailedMeasurementReading failedMeasurementReading );

    /**
     * Fetch multiple
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MeasurementDocument
     * MeasurementDocument} entities.
     * 
     * @return A request builder to fetch multiple
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MeasurementDocument
     *         MeasurementDocument} entities. This request builder allows methods which modify the underlying query to
     *         be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MeasurementDocument>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetAllRequestBuilder<MeasurementDocument> getAllMeasurementDocument();

    /**
     * Fetch the number of entries from the
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MeasurementDocument
     * MeasurementDocument} entity collection matching the filter and search expressions.
     * 
     * @return A request builder to fetch the count of
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MeasurementDocument
     *         MeasurementDocument} entities. This request builder allows methods which modify the underlying query to
     *         be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MeasurementDocument>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CountRequestBuilder<MeasurementDocument> countMeasurementDocument();

    /**
     * Fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MeasurementDocument
     * MeasurementDocument} entity using key fields.
     * 
     * @param measurementDocument
     *            Measurement Document
     *            <p>
     *            Constraints: Not nullable, Maximum length: 20
     *            </p>
     * @return A request builder to fetch a single
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MeasurementDocument
     *         MeasurementDocument} entity using key fields. This request builder allows methods which modify the
     *         underlying query to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MeasurementDocument>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetByKeyRequestBuilder<MeasurementDocument>
        getMeasurementDocumentByKey( @Nonnull final String measurementDocument );

    /**
     * Create a new {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MeasurementDocument
     * MeasurementDocument} entity and save it to the S/4HANA system.
     * 
     * @param measurementDocument
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MeasurementDocument
     *            MeasurementDocument} entity object that will be created in the S/4HANA system.
     * @return A request builder to create a new
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MeasurementDocument
     *         MeasurementDocument} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MeasurementDocument>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CreateRequestBuilder<MeasurementDocument>
        createMeasurementDocument( @Nonnull final MeasurementDocument measurementDocument );

    /**
     * Update an existing
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MeasurementDocument
     * MeasurementDocument} entity and save it to the S/4HANA system.
     * 
     * @param measurementDocument
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MeasurementDocument
     *            MeasurementDocument} entity object that will be updated in the S/4HANA system.
     * @return A request builder to update an existing
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MeasurementDocument
     *         MeasurementDocument} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MeasurementDocument>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    UpdateRequestBuilder<MeasurementDocument>
        updateMeasurementDocument( @Nonnull final MeasurementDocument measurementDocument );

    /**
     * Fetch multiple
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MsmtDocumentLongText
     * MsmtDocumentLongText} entities.
     * 
     * @return A request builder to fetch multiple
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MsmtDocumentLongText
     *         MsmtDocumentLongText} entities. This request builder allows methods which modify the underlying query to
     *         be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MsmtDocumentLongText>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetAllRequestBuilder<MsmtDocumentLongText> getAllMsmtDocumentLongText();

    /**
     * Fetch the number of entries from the
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MsmtDocumentLongText
     * MsmtDocumentLongText} entity collection matching the filter and search expressions.
     * 
     * @return A request builder to fetch the count of
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MsmtDocumentLongText
     *         MsmtDocumentLongText} entities. This request builder allows methods which modify the underlying query to
     *         be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MsmtDocumentLongText>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CountRequestBuilder<MsmtDocumentLongText> countMsmtDocumentLongText();

    /**
     * Fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MsmtDocumentLongText
     * MsmtDocumentLongText} entity using key fields.
     * 
     * @param measurementDocument
     *            Measurement Document
     *            <p>
     *            Constraints: Not nullable, Maximum length: 20
     *            </p>
     * @return A request builder to fetch a single
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MsmtDocumentLongText
     *         MsmtDocumentLongText} entity using key fields. This request builder allows methods which modify the
     *         underlying query to be called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MsmtDocumentLongText>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetByKeyRequestBuilder<MsmtDocumentLongText>
        getMsmtDocumentLongTextByKey( @Nonnull final String measurementDocument );

    /**
     * Create a new
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MsmtDocumentLongText
     * MsmtDocumentLongText} entity and save it to the S/4HANA system.
     * 
     * @param msmtDocumentLongText
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MsmtDocumentLongText
     *            MsmtDocumentLongText} entity object that will be created in the S/4HANA system.
     * @return A request builder to create a new
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MsmtDocumentLongText
     *         MsmtDocumentLongText} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MsmtDocumentLongText>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CreateRequestBuilder<MsmtDocumentLongText>
        createMsmtDocumentLongText( @Nonnull final MsmtDocumentLongText msmtDocumentLongText );

    /**
     * Update an existing
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MsmtDocumentLongText
     * MsmtDocumentLongText} entity and save it to the S/4HANA system.
     * 
     * @param msmtDocumentLongText
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MsmtDocumentLongText
     *            MsmtDocumentLongText} entity object that will be updated in the S/4HANA system.
     * @return A request builder to update an existing
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MsmtDocumentLongText
     *         MsmtDocumentLongText} entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.measurementdocument.MsmtDocumentLongText>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    UpdateRequestBuilder<MsmtDocumentLongText>
        updateMsmtDocumentLongText( @Nonnull final MsmtDocumentLongText msmtDocumentLongText );

}
