/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odatav4.services;

import java.util.UUID;

import javax.annotation.Nonnull;

import com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder;
import com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder;
import com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.supplieritem.SupplierItem;

/**
 * <p>
 * You can use this service to create, read and update supplier item. The service is based on the OData protocol and can
 * be consumed by external systems.
 * </p>
 * <p>
 * <a href=
 * 'https://help.sap.com:00443/http.svc/ahp2/SAP_S4HANA_CLOUD/latest/EN/74/c6b169bfb74953b07a66aa248cd9aa/frameset.htm'>Business
 * Documentation</a>
 * </p>
 * <p>
 * Reference:
 * <a href='https://api.sap.com/shell/discover/contentpackage/SAPS4HANACloud/api/SUPPLIERITEM_0001?section=OVERVIEW'>SAP
 * API Business Hub</a>
 * </p>
 * <h3>Details:</h3>
 * <table summary='Details'>
 * <tr>
 * <td align='right'>OData Service:</td>
 * <td>SUPPLIERITEM_0001</td>
 * </tr>
 * <tr>
 * <td align='right'>API Version:</td>
 * <td>1</td>
 * </tr>
 * <tr>
 * <td align='right'>Communication Scenario:</td>
 * <td>Field Logistics - Manage Supplier Item Integration (SAP_COM_0732)</td>
 * </tr>
 * <tr>
 * <td align='right'>Scope Items:</td>
 * <td><a href='https://rapid.sap.com/bp/#/scopeitems/4AH'>Field Logistics Planning and Execution – Supplier Items
 * (4AH)</a></td>
 * </tr>
 * <tr>
 * <td align='right'>Authentication Methods:</td>
 * <td>Basic, x509</td>
 * </tr>
 * <tr>
 * <td align='right'>Service Group Name:</td>
 * <td>API_SUPPLIERITEM</td>
 * </tr>
 * <tr>
 * <td align='right'>Business Object:</td>
 * <td>FldLogsSupplierItem</td>
 * </tr>
 * </table>
 * 
 */
public interface SupplierItemService
{

    /**
     * If no other path was provided via the {@link #withServicePath(String)} method, this is the default service path
     * used to access the endpoint.
     * 
     */
    String DEFAULT_SERVICE_PATH = "/sap/opu/odata4/sap/api_supplieritem/srvd_a2x/sap/supplieritem/0001";

    /**
     * Overrides the default service path and returns a new service instance with the specified service path. Also
     * adjusts the respective entity URLs.
     * 
     * @param servicePath
     *            Service path that will override the default.
     * @return A new service instance with the specified service path.
     */
    @Nonnull
    SupplierItemService withServicePath( @Nonnull final String servicePath );

    /**
     * Creates a batch request builder object.
     * 
     * @return A request builder to handle batch operation on this service. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.BatchRequestBuilder#execute(HttpDestinationProperties)
     *         execute} method on the request builder object.
     */
    @Nonnull
    BatchRequestBuilder batch();

    /**
     * Fetch multiple {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.supplieritem.SupplierItem
     * SupplierItem} entities.
     * 
     * @return A request builder to fetch multiple
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.supplieritem.SupplierItem SupplierItem}
     *         entities. This request builder allows methods which modify the underlying query to be called before
     *         executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetAllRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.supplieritem.SupplierItem>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetAllRequestBuilder<SupplierItem> getAllSupplierItem();

    /**
     * Fetch the number of entries from the
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.supplieritem.SupplierItem SupplierItem} entity
     * collection matching the filter and search expressions.
     * 
     * @return A request builder to fetch the count of
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.supplieritem.SupplierItem SupplierItem}
     *         entities. This request builder allows methods which modify the underlying query to be called before
     *         executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CountRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.supplieritem.SupplierItem>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CountRequestBuilder<SupplierItem> countSupplierItem();

    /**
     * Fetch a single {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.supplieritem.SupplierItem
     * SupplierItem} entity using key fields.
     * 
     * @param fldLogsSuplrItemUUID
     *            Supplier Item UUID
     *            <p>
     *            Constraints: Not nullable
     *            </p>
     * @return A request builder to fetch a single
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.supplieritem.SupplierItem SupplierItem}
     *         entity using key fields. This request builder allows methods which modify the underlying query to be
     *         called before executing the query itself. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.GetByKeyRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.supplieritem.SupplierItem>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    GetByKeyRequestBuilder<SupplierItem> getSupplierItemByKey( @Nonnull final UUID fldLogsSuplrItemUUID );

    /**
     * Create a new {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.supplieritem.SupplierItem SupplierItem}
     * entity and save it to the S/4HANA system.
     * 
     * @param supplierItem
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.supplieritem.SupplierItem SupplierItem}
     *            entity object that will be created in the S/4HANA system.
     * @return A request builder to create a new
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.supplieritem.SupplierItem SupplierItem}
     *         entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.CreateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.supplieritem.SupplierItem>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    CreateRequestBuilder<SupplierItem> createSupplierItem( @Nonnull final SupplierItem supplierItem );

    /**
     * Update an existing {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.supplieritem.SupplierItem
     * SupplierItem} entity and save it to the S/4HANA system.
     * 
     * @param supplierItem
     *            {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.supplieritem.SupplierItem SupplierItem}
     *            entity object that will be updated in the S/4HANA system.
     * @return A request builder to update an existing
     *         {@link com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.supplieritem.SupplierItem SupplierItem}
     *         entity. To perform execution, call the
     *         {@link com.sap.cloud.sdk.datamodel.odatav4.core.UpdateRequestBuilder<com.sap.cloud.sdk.s4hana.datamodel.odatav4.namespaces.supplieritem.SupplierItem>#execute
     *         execute} method on the request builder object.
     */
    @Nonnull
    UpdateRequestBuilder<SupplierItem> updateSupplierItem( @Nonnull final SupplierItem supplierItem );

}
