package com.zbank.file.common.http;

import java.net.URLDecoder;
import java.nio.charset.Charset;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSession;

import org.apache.commons.lang3.StringUtils;
import org.apache.http.Consts;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHost;
import org.apache.http.HttpStatus;
import org.apache.http.NameValuePair;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.utils.HttpClientUtils;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.HttpClientConnectionManager;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.entity.ContentType;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.ssl.SSLContexts;
import org.apache.http.ssl.TrustStrategy;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.zbank.file.common.http.bean.BinaryDataRequest;
import com.zbank.file.common.http.bean.BinaryDataResponse;
import com.zbank.file.common.http.bean.Request;
import com.zbank.file.common.http.bean.Response;
import com.zbank.file.common.http.config.HttpConfig;
import com.zbank.file.common.http.ss5.HttpClientFactory;
import com.zbank.file.common.utils.JsonUtil;
import com.zbank.file.exception.SDKException;

/**
 * http请求调用
 * 
 * @author zhanglulu
 *
 */
public class HttpClientUtil {

	private static final Logger log = LoggerFactory.getLogger(HttpClientUtil.class);
	
	/**
	 * 以流的形式下载文件
	 * 
	 * @param request
	 * @return
	 * @throws SDKException
	 */
	public static BinaryDataResponse downloadInputStream(Request request) throws SDKException {
		return downloadInputStream(request, HttpConfig.defaultConfig);
	}

	/**
	 * 以流的形式下载文件
	 * 
	 * @param request
	 * @return
	 * @throws SDKException
	 */
	public static BinaryDataResponse downloadInputStream(Request request, HttpConfig conf) throws SDKException {
		CloseableHttpResponse resp = null;
		CloseableHttpClient client = null;
		try {
			if(conf == null) {
				conf = HttpConfig.defaultConfig;
			}
			if (conf.isSockts()) {
				client = HttpClientFactory.getInstance().getHttpClient(conf);
			} else {
				client = createHttpClient(conf);
			}
			resp = interPost(request, conf,client);
			Header header = resp.getFirstHeader("resultMap");
			//header内容解密
			String headerData = URLDecoder.decode(header.getValue(), "UTF-8");
			log.info("返回的结果是：{}", headerData);
			Map<String, Object> map = JsonUtil.MAPPER.readValue(headerData, HashMap.class);
			return new BinaryDataResponse(map.get("code") == null ? "" : map.get("code").toString(),
					map.get("msg") == null ? "" : map.get("msg").toString(),resp,client, map);
		} catch (Exception e) {
			HttpClientUtils.closeQuietly(resp);
			HttpClientUtils.closeQuietly(client);
			if( e instanceof SDKException){
				throw (SDKException)e;
			}
			log.error("请求发送异常", e);
			throw new SDKException("请求发送异常", e);
		} 
	}

	/**
	 * 以二进制字节数字的形式下载文件
	 * 
	 * @param request
	 * @return
	 * @throws SDKException
	 */
	public static BinaryDataResponse downloadBinaryData(Request request) throws SDKException {
		return downloadBinaryData(request, HttpConfig.defaultConfig);
	}

	/**
	 * 以二进制字节数字的形式下载文件
	 * 
	 * @param request
	 * @return
	 * @throws SDKException
	 */
	public static BinaryDataResponse downloadBinaryData(Request request, HttpConfig conf) throws SDKException {
		CloseableHttpResponse resp = null;
		CloseableHttpClient client = null;
		try {
			if(conf == null) {
				conf = HttpConfig.defaultConfig;
			}
			if (conf.isSockts()) {
				client = HttpClientFactory.getInstance().getHttpClient(conf);
			} else {
				client = createHttpClient(conf);
			}
			resp = interPost(request, conf,client);
			HttpEntity entity = resp.getEntity();
			Header header = resp.getFirstHeader("resultMap");
			//header内容解密
			String headerData = URLDecoder.decode(header.getValue(), "UTF-8");
			log.info("返回的结果是：{}", headerData);
			Map<String, Object> map = JsonUtil.MAPPER.readValue(headerData, HashMap.class);
			return new BinaryDataResponse(map.get("code") == null ? "" : map.get("code").toString(),
					map.get("msg") == null ? "" : map.get("msg").toString(), EntityUtils.toByteArray(entity), map);
		} catch (Exception e) {
			if( e instanceof SDKException){
				throw (SDKException)e;
			}
			log.error("请求发送异常", e);
			throw new SDKException("请求发送异常", e);
		} finally {
			HttpClientUtils.closeQuietly(resp);
			HttpClientUtils.closeQuietly(client);
		}
	}

	/**
	 * 以二进制字节数字的形式上传文件
	 * 
	 * @param request
	 * @return
	 * @throws SDKException
	 */
	public static Response uploadBinaryData(BinaryDataRequest request) throws SDKException {
		return uploadBinaryData(request, HttpConfig.defaultConfig);
	}

	/**
	 * 以二进制字节数组的形式上传文件
	 * 
	 * @param request
	 * @return
	 * @throws SDKException
	 */
	public static Response uploadBinaryData(BinaryDataRequest request, HttpConfig conf) throws SDKException {

		log.info("invoking url[" + request.getUrl() + "]service[" + request.getServiceId() + "]");
		CloseableHttpResponse resp = null;
		CloseableHttpClient client = null;
		try {
			if(conf == null) {
				conf = HttpConfig.defaultConfig;
			}
			if (conf.isSockts()) {
				client = HttpClientFactory.getInstance().getHttpClient(conf);
			} else {
				client = createHttpClient(conf);
			}
			HttpPost httpPost = new HttpPost(request.getUrl());
			RequestConfig.Builder builder = RequestConfig.custom();

			if (conf.getConnRequestTimeout() != -1) {
				builder.setConnectionRequestTimeout(conf.getConnRequestTimeout());
			}
			if (conf.getConnTimeout() != -1) {
				builder.setConnectTimeout(conf.getConnTimeout());
			}
			if (conf.getSocketTimeout() != -1) {
				builder.setSocketTimeout(conf.getSocketTimeout());
			}
			if (StringUtils.isNotBlank(conf.getProxyHost()) && -1 != conf.getProxyPort()) {
				HttpHost host = new HttpHost(conf.getProxyHost(), conf.getProxyPort());
				builder.setProxy(host);
			}
			httpPost.setConfig(builder.build());
			/**
			 * 文件网关无法接收filepart，该方式不可用
			 * MultipartEntityBuilder builder = MultipartEntityBuilder.create(); 
			 * builder.setCharset(Consts.UTF_8);
			   builder.setMode(HttpMultipartMode.BROWSER_COMPATIBLE); 
			   builder.addBinaryBody("file", request.getBinaryData(), ContentType.MULTIPART_FORM_DATA, "sections");
			 * 
			 */
			httpPost.setEntity(new ByteArrayEntity(request.getBinaryData()));
			if (request.getHeaderData() != null) {
				Iterator<Entry<String, String>> ite = request.getHeaderData().entrySet().iterator();
				while (ite.hasNext()) {
					Entry<String, String> entry = ite.next();
					httpPost.addHeader(entry.getKey(), entry.getValue());
				}
			}
			resp = client.execute(httpPost);

			if(resp.getStatusLine().getStatusCode() != HttpStatus.SC_OK){
				log.error("http response status error: {}", resp.getStatusLine());
			}
			String response = EntityUtils.toString(resp.getEntity(), Charset.forName("UTF-8"));
			log.info("http请求响应的结果是:" + response);
			Map<String, Object> map = JsonUtil.MAPPER.readValue(response, HashMap.class);

			return new Response(map);
		} catch (Exception e) {
			log.error("请求发送异常", e);
			throw new SDKException("请求发送异常", e);
		} finally {
			HttpClientUtils.closeQuietly(resp);
			HttpClientUtils.closeQuietly(client);
		}

	}

	/**
	 * 普通的http请求调用，可以用于一般的接口调用
	 * 
	 * @param param
	 * @return
	 * @throws SDKException
	 */
	public static Response post(Request param) throws SDKException {
		return post(param, HttpConfig.defaultConfig);
	}

	/**
	 * 普通的http请求调用，可以用于一般的接口调用
	 * 
	 * @param
	 * @return
	 * @throws SDKException
	 */
	public static Response post(Request request, HttpConfig conf) throws SDKException {
		CloseableHttpResponse resp = null;
		CloseableHttpClient client = null;
		try {
			if(conf == null) {
				conf = HttpConfig.defaultConfig;
			}
			if (conf.isSockts()) {
				client = HttpClientFactory.getInstance().getHttpClient(conf);
			} else {
				client = createHttpClient(conf);
			}
			resp = interPost(request, conf,client);
			HttpEntity httpEntity = resp.getEntity();
			String entity = EntityUtils.toString(httpEntity, Charset.forName("UTF-8"));
			log.info("http请求响应的结果是:" + entity);
			Map<String, Object> map = JsonUtil.MAPPER.readValue(entity, HashMap.class);
			return new Response(map);
		} catch (Exception e) {
			if( e instanceof SDKException){
				throw (SDKException)e;
			}
			log.error("请求发送异常", e);
			throw new SDKException("请求发送异常", e);
		} finally {
			HttpClientUtils.closeQuietly(resp);
			HttpClientUtils.closeQuietly(client);
		}
	}
	
	private static CloseableHttpResponse interPost(Request request, HttpConfig conf,CloseableHttpClient client) throws SDKException {
		log.info("invoking url[" + request.getUrl() + "]service[" + request.getServiceId() + "]");
		CloseableHttpResponse resp = null;
		try {

			HttpPost httpPost = new HttpPost(request.getUrl());
			RequestConfig.Builder builder = RequestConfig.custom();

			if (conf.getConnRequestTimeout() != -1) {
				builder.setConnectionRequestTimeout(conf.getConnRequestTimeout());
			}
			if (conf.getConnTimeout() != -1) {
				builder.setConnectTimeout(conf.getConnTimeout());
			}
			if (conf.getSocketTimeout() != -1) {
				builder.setSocketTimeout(conf.getSocketTimeout());
			}
			if (StringUtils.isNotBlank(conf.getProxyHost()) && -1 != conf.getProxyPort()) {
				HttpHost host = new HttpHost(conf.getProxyHost(), conf.getProxyPort());
				builder.setProxy(host);
			}
			if (request.getHeaderData() != null) {
				Iterator<Entry<String, String>> ite = request.getHeaderData().entrySet().iterator();
				while (ite.hasNext()) {
					Entry<String, String> entry = ite.next();
					httpPost.addHeader(entry.getKey(), entry.getValue());
				}
			}
			httpPost.setConfig(builder.build());
			if(request.getEntityData() != null){
				// 创建参数队列
				List<NameValuePair> formParamList = new ArrayList<NameValuePair>();
				Iterator<Entry<String, String>> it = request.getEntityData().entrySet().iterator();
				while (it.hasNext()) {
					Entry<String, String> entry = it.next();
					formParamList.add(new BasicNameValuePair(entry.getKey(), entry.getValue()==null?"":entry.getValue().toString()));
				}
				UrlEncodedFormEntity stringEntity = new UrlEncodedFormEntity(formParamList, Consts.UTF_8);
				stringEntity.setContentType(ContentType.APPLICATION_FORM_URLENCODED.toString());
				httpPost.setEntity(stringEntity);
			}
			resp = client.execute(httpPost);

			if(resp.getStatusLine().getStatusCode() != HttpStatus.SC_OK){
				log.error("http response status error:" + resp.getStatusLine());
			}
			return resp;
		} catch (Exception e) {
			log.error("请求发送异常", e);
			throw new SDKException("请求发送异常", e);
		}
	}

	/**
	 * 创建httpsclient
	 */
	private static CloseableHttpClient createHttpClient(HttpConfig conf) {
		// HttpClientUtils.closeQuietly(client);
		// Locale.setDefault(new Locale("zh","CN"));
		HttpClientBuilder builder = HttpClientBuilder.create();
		// BasicHttpClientConnectionManager connMan = new
		// BasicHttpClientConnectionManager();
		builder.setConnectionManager(init(conf));
		// builder.setRetryHandler(new MyHttpRequestRetryHandler());
		// builder.setDefaultHeaders(getDefaultHeaders());
		// builder.setSSLHostnameVerifier(SSLConnectionSocketFactory.getDefaultHostnameVerifier());
		return builder.build();
	}

	private static HttpClientConnectionManager init(HttpConfig conf) {
		try {
			SSLContext sslContext = SSLContexts.custom().loadTrustMaterial(null, new TrustStrategy() {

				@Override
				public boolean isTrusted(X509Certificate[] arg0, String arg1) throws java.security.cert.CertificateException {
					return true;
				}

			}).build();
			SSLConnectionSocketFactory sslsf = new SSLConnectionSocketFactory(sslContext, new String[] {"TLSv1","TLSv1.1","TLSv1.2"}, null,
					SSLConnectionSocketFactory.getDefaultHostnameVerifier());
//			SSLConnectionSocketFactory sslsf = new SSLConnectionSocketFactory(sslContext, new String[] {"TLSv1","TLSv1.1","TLSv1.2"}, null,
//					new HostnameVerifier(){
//						@Override
//						public boolean verify(String hostname, SSLSession session) {
//							hostname = "file.z-bank.com";//适配紫金农商，非通用
//							return SSLConnectionSocketFactory.getDefaultHostnameVerifier().verify(hostname, session);
//						}
//
//					}
//				);
			Registry<ConnectionSocketFactory> registry = RegistryBuilder.<ConnectionSocketFactory> create()
					.register("http", PlainConnectionSocketFactory.INSTANCE).register("https", sslsf).build();
			PoolingHttpClientConnectionManager mg = new PoolingHttpClientConnectionManager(registry);
			if (conf.getMaxConnPerRoute() != -1) {
				mg.setDefaultMaxPerRoute(conf.getMaxConnPerRoute());
			}
			if (conf.getMaxConnTotal() != -1) {
				mg.setMaxTotal(conf.getMaxConnTotal());
			}
			return mg;
		} catch (Exception e) {
			log.error("客户端网络初始化异常", e);
			e.printStackTrace();
		}
		return null;
	}
}
